#include "GamebaseBridgePurchase.h"
#include "GamebaseCommunicationRequestTypes.h"
#include "GamebaseCommunicator.h"
#include "GamebaseCommunicatorAsyncSender.h"

namespace GamebaseScheme
{
    const FName RequestPurchaseSeq(TEXT("gamebase://requestPurchaseSeq"));
    const FName RequestPurchaseProductId(TEXT("gamebase://requestPurchaseProductId"));
    const FName RequestPurchaseProductIdWithPayload(TEXT("gamebase://requestPurchaseProductIdWithPayload"));
    const FName RequestItemListOfNotConsumed(TEXT("gamebase://requestItemListOfNotConsumed"));
    const FName RequestItemListPurchasable(TEXT("gamebase://requestItemListPurchasable"));
    const FName RequestItemListAtIAPConsole(TEXT("gamebase://requestItemListAtIAPConsole"));
    const FName SetPromotionIAPHandler(TEXT("gamebase://setPromotionIAPHandler"));
    const FName RequestActivatedPurchases(TEXT("gamebase://requestActivatedPurchases"));
}


namespace GamebaseCommunicatorAsyncSender
{
    void Purchase(const FGamebaseCommunicatorSendData& SendData, FGamebaseCommunicatorManager* Communicator, const FGamebasePurchasableReceiptDelegate& Callback)
    {
        Communicator->RegisterAsyncEvent(SendData, [=](const FGamebaseCommunicatorReceiveDataRef& ReceiveData)
        {
            const auto Error = GetGamebaseError(ReceiveData);
            if (ReceiveData->JsonData.IsEmpty())
            {
                Callback.ExecuteIfBound(nullptr, Error.Get());
                return;
            }
            
            const FGamebasePurchasableReceiptPtr Receipt = MakeShared<FGamebasePurchasableReceipt, ESPMode::ThreadSafe>();
            Receipt->FromJson(ReceiveData->JsonData);
            
            Callback.ExecuteIfBound(Receipt.Get(), Error.Get());
        });
    }
    
    void RequestItemListOfNotConsumed(const FGamebaseCommunicatorSendData& SendData, FGamebaseCommunicatorManager* Communicator, const FGamebasePurchasableReceiptListDelegate& Callback)
    {
        Communicator->RegisterAsyncEvent(SendData, [=](const FGamebaseCommunicatorReceiveDataRef& ReceiveData)
        {
            const auto Error = GetGamebaseError(ReceiveData);
            if (ReceiveData->JsonData.IsEmpty())
            {
                Callback.ExecuteIfBound(nullptr, Error.Get());
                return;
            }

            const TSharedPtr<TArray<FGamebasePurchasableReceipt>> ResultList = MakeShared<TArray<FGamebasePurchasableReceipt>>(
                GamebaseJsonUtils::ConvertJsonStringToJsonArray<FGamebasePurchasableReceipt>(ReceiveData->JsonData));
            Callback.ExecuteIfBound(ResultList.Get(), Error.Get());
        });
    }

    void ItemList(const FGamebaseCommunicatorSendData& SendData, FGamebaseCommunicatorManager* Communicator, const FGamebasePurchasableItemListDelegate& Callback)
    {
        Communicator->RegisterAsyncEvent(SendData, [=](const FGamebaseCommunicatorReceiveDataRef& ReceiveData)
        {
            const auto Error = GetGamebaseError(ReceiveData);
            if (ReceiveData->JsonData.IsEmpty())
            {
                Callback.ExecuteIfBound(nullptr, Error.Get());
                return;
            }
            
            const TSharedPtr<TArray<FGamebasePurchasableItem>> ResultList = MakeShared<TArray<FGamebasePurchasableItem>>(
                GamebaseJsonUtils::ConvertJsonStringToJsonArray<FGamebasePurchasableItem>(ReceiveData->JsonData));
            Callback.ExecuteIfBound(ResultList.Get(), Error.Get());
        });
    }
}

FGamebaseBridgePurchase::FGamebaseBridgePurchase(const TSharedPtr<FGamebaseCommunicatorManager>& Communicator, const FName& ClassName)
    : FGamebaseBridgeBase(Communicator, ClassName)
{
}

void FGamebaseBridgePurchase::RequestPurchase(const int64 ItemSeq, const FGamebasePurchasableReceiptDelegate& Callback)
{
    FPurchaseItemSeq PurchaseItemSeq;
    PurchaseItemSeq.ItemSeq = ItemSeq;
    
    FGamebaseCommunicatorSendData Data { GamebaseScheme::RequestPurchaseSeq };
    Data.JsonData = PurchaseItemSeq.ToJson(false);

    GamebaseCommunicatorAsyncSender::Purchase(Data, Communicator.Get(), Callback);
}

void FGamebaseBridgePurchase::RequestPurchase(const FString& GamebaseProductId, const FGamebasePurchasableReceiptDelegate& Callback)
{
    FPurchaseProductId PurchaseProductId;
    PurchaseProductId.GamebaseProductId = GamebaseProductId;
    
    FGamebaseCommunicatorSendData Data { GamebaseScheme::RequestPurchaseProductId };
    Data.JsonData = PurchaseProductId.ToJson(false);

    GamebaseCommunicatorAsyncSender::Purchase(Data, Communicator.Get(), Callback);
}

void FGamebaseBridgePurchase::RequestPurchase(const FString& GamebaseProductId, const FString& Payload, const FGamebasePurchasableReceiptDelegate& Callback)
{
    FPurchaseProductId PurchaseProductId;
    PurchaseProductId.GamebaseProductId = GamebaseProductId;
    PurchaseProductId.Payload = Payload;
    
    FGamebaseCommunicatorSendData Data { GamebaseScheme::RequestPurchaseProductIdWithPayload };
    Data.JsonData = PurchaseProductId.ToJson(false);

    GamebaseCommunicatorAsyncSender::Purchase(Data, Communicator.Get(), Callback);
}

void FGamebaseBridgePurchase::RequestItemListOfNotConsumed(const FGamebasePurchasableReceiptListDelegate& Callback)
{
    const FGamebasePurchasableConfiguration Configuration;
    RequestItemListOfNotConsumed(Configuration, Callback);
}

void FGamebaseBridgePurchase::RequestItemListOfNotConsumed(const FGamebasePurchasableConfiguration& Configuration, const FGamebasePurchasableReceiptListDelegate& Callback)
{
    FGamebaseCommunicatorSendData Data{ GamebaseScheme::RequestItemListOfNotConsumed };
    Data.JsonData = Configuration.ToJson();
    Data.ExtraData = Configuration.ToJson();           //TODO: extra configuration 필요한지 확인 필요
    
    GamebaseCommunicatorAsyncSender::RequestItemListOfNotConsumed(Data, Communicator.Get(), Callback);
}

void FGamebaseBridgePurchase::RequestItemListPurchasable(const FGamebasePurchasableItemListDelegate& Callback)
{
    const FGamebaseCommunicatorSendData Data{ GamebaseScheme::RequestItemListPurchasable };

    GamebaseCommunicatorAsyncSender::ItemList(Data, Communicator.Get(), Callback);
}

void FGamebaseBridgePurchase::RequestItemListAtIAPConsole(const FGamebasePurchasableItemListDelegate& Callback)
{
    const FGamebaseCommunicatorSendData Data{ GamebaseScheme::RequestItemListAtIAPConsole };

    GamebaseCommunicatorAsyncSender::ItemList(Data, Communicator.Get(), Callback);
}

void FGamebaseBridgePurchase::SetPromotionIAPHandler(const FGamebasePurchasableReceiptDelegate& Callback)
{
    const FGamebaseCommunicatorSendData Data{ GamebaseScheme::SetPromotionIAPHandler };

    GamebaseCommunicatorAsyncSender::Purchase(Data, Communicator.Get(), Callback);
}

void FGamebaseBridgePurchase::RequestActivatedPurchases(const FGamebasePurchasableReceiptListDelegate& Callback)
{
    const FGamebasePurchasableConfiguration Configuration;
    RequestActivatedPurchases(Configuration, Callback);
}

void FGamebaseBridgePurchase::RequestActivatedPurchases(const FGamebasePurchasableConfiguration& Configuration, const FGamebasePurchasableReceiptListDelegate& Callback)
{
    FGamebaseCommunicatorSendData Data{ GamebaseScheme::RequestActivatedPurchases };
    Data.JsonData = Configuration.ToJson();
    Data.ExtraData = Configuration.ToJson();           //TODO: extra configuration 필요한지 확인 필요
    
    GamebaseCommunicatorAsyncSender::RequestItemListOfNotConsumed(Data, Communicator.Get(), Callback);
}

void FGamebaseBridgePurchase::RequestSubscriptionsStatus(const FGamebasePurchasableConfiguration& Configuration, const FGamebasePurchasableSubscriptionStatusDelegate& Callback)
{
}
