#include "GamebaseBridgeContact.h"
#include "GamebaseCommunicator.h"
#include "GamebaseCommunicatorAsyncSender.h"
#include "GamebaseCommunicatorManager.h"

namespace GamebaseScheme
{
    const FName OpenContact(TEXT("gamebase://openContact"));
    const FName OpenContactWithConfiguration(TEXT("gamebase://openContactWithConfiguration"));
    const FName RequestContactURL(TEXT("gamebase://requestContactURL"));
    const FName RequestContactURLWithConfiguration(TEXT("gamebase://requestContactURLWithConfiguration"));
}

namespace GamebaseCommunicatorAsyncSender
{
    void OpenContact(const FGamebaseCommunicatorSendData& SendData, FGamebaseCommunicatorManager* Communicator, const FGamebaseErrorDelegate& Callback)
    {
        Communicator->RegisterAsyncEvent(SendData, [=](const FGamebaseCommunicatorReceiveDataRef& ReceiveData)
        {
            Callback.ExecuteIfBound(GetGamebaseError(ReceiveData).Get());
        });
    }

    void RequestContactURL(const FGamebaseCommunicatorSendData& SendData, FGamebaseCommunicatorManager* Communicator, const FGamebaseContactUrlDelegate& Callback)
    {
        Communicator->RegisterAsyncEvent(SendData, [=](const FGamebaseCommunicatorReceiveDataRef& ReceiveData)
        {
            const auto Error = GetGamebaseError(ReceiveData);
            if (ReceiveData->JsonData.IsEmpty())
            {
                Callback.ExecuteIfBound(FString(), Error.Get());
                return;
            }

            Callback.ExecuteIfBound(ReceiveData->JsonData, Error.Get());
        });
    }
}

FGamebaseBridgeContact::FGamebaseBridgeContact(const TSharedPtr<FGamebaseCommunicatorManager>& Communicator, const FName& ClassName)
    : FGamebaseBridgeBase(Communicator, ClassName)
{
}

void FGamebaseBridgeContact::OpenContact(const FGamebaseErrorDelegate& CloseCallback)
{
    FGamebaseCommunicatorSendData SendData { GamebaseScheme::OpenContact };

    GamebaseCommunicatorAsyncSender::OpenContact(SendData, Communicator.Get(), CloseCallback);
}

void FGamebaseBridgeContact::OpenContact(const FGamebaseContactConfiguration& Configuration, const FGamebaseErrorDelegate& CloseCallback)
{
    FGamebaseCommunicatorSendData SendData{ GamebaseScheme::OpenContactWithConfiguration };
    SendData.JsonData = Configuration.ToJson();

    GamebaseCommunicatorAsyncSender::OpenContact(SendData, Communicator.Get(), CloseCallback);
}

void FGamebaseBridgeContact::RequestContactURL(const FGamebaseContactUrlDelegate& Callback)
{
    FGamebaseCommunicatorSendData SendData{ GamebaseScheme::RequestContactURL };

    GamebaseCommunicatorAsyncSender::RequestContactURL(SendData, Communicator.Get(), Callback);
}

void FGamebaseBridgeContact::RequestContactURL(const FGamebaseContactConfiguration& Configuration, const FGamebaseContactUrlDelegate& Callback)
{
    FGamebaseCommunicatorSendData SendData{ GamebaseScheme::RequestContactURLWithConfiguration };
    SendData.JsonData = Configuration.ToJson();

    GamebaseCommunicatorAsyncSender::RequestContactURL(SendData, Communicator.Get(), Callback);
}
