#include "GamebaseBridgeAuth.h"

#include "GamebaseBridgeAuthAdapter.h"
#include "GamebaseBridgeSubsystem.h"
#include "GamebaseCommunicationRequestTypes.h"
#include "GamebaseCommunicatorAsyncSender.h"
#include "GamebaseCommunicatorManager.h"
#include "GamebaseDebugLogger.h"

namespace GamebaseScheme
{
    const FName Login(TEXT("gamebase://login"));
    const FName LoginWithAdditionalInfo(TEXT("gamebase://loginWithAdditionalInfo"));
    const FName LoginWithCredentialInfo(TEXT("gamebase://loginWithCredentialInfo"));
    const FName LoginForLastLoggedInProvider(TEXT("gamebase://loginForLastLoggedInProvider"));
    const FName LoginForLastLoggedInProviderWithAdditionalInfo(TEXT("gamebase://loginForLastLoggedInProviderWithAdditionalInfo"));
    const FName Logout(TEXT("gamebase://logout"));

    const FName Withdraw(TEXT("gamebase://withdraw"));
    const FName WithdrawImmediately(TEXT("gamebase://withdrawImmediately"));
    const FName RequestTemporaryWithdrawal(TEXT("gamebase://requestTemporaryWithdrawal"));
    const FName CancelTemporaryWithdrawal(TEXT("gamebase://cancelTemporaryWithdrawal"));

    const FName AddMapping(TEXT("gamebase://addMapping"));
    const FName AddMappingWithCredentialInfo(TEXT("gamebase://addMappingWithCredentialInfo"));
    const FName AddMappingWithAdditionalInfo(TEXT("gamebase://addMappingWithAdditionalInfo"));

    const FName ChangeLogin(TEXT("gamebase://changeLoginWithForcingMappingTicket"));
    const FName AddMappingForciblyWithForcingMappingTicket(TEXT("gamebase://addMappingForciblyWithForcingMappingTicket"));
    
    const FName AddMappingForcibly(TEXT("gamebase://addMappingForcibly"));
    const FName AddMappingForciblyWithCredentialInfo(TEXT("gamebase://addMappingForciblyWithCredentialInfo"));
    const FName AddMappingForciblyWithAdditionalInfo(TEXT("gamebase://addMappingForciblyWithAdditionalInfo"));
    const FName RemoveMapping(TEXT("gamebase://removeMapping"));

    const FName IssueTransferAccount(TEXT("gamebase://issueTransferAccount"));
    const FName QueryTransferAccount(TEXT("gamebase://queryTransferAccount"));
    const FName RenewTransferAccount(TEXT("gamebase://renewTransferAccount"));
    const FName TransferAccountWithIdPLogin(TEXT("gamebase://transferAccountWithIdPLogin"));

    const FName GetAuthMappingList(TEXT("gamebase://getAuthMappingList"));
    const FName GetAuthProviderUserID(TEXT("gamebase://getAuthProviderUserID"));
    const FName GetAuthProviderAccessToken(TEXT("gamebase://getAuthProviderAccessToken"));
    const FName GetAuthProviderProfile(TEXT("gamebase://getAuthProviderProfile"));
    const FName GetBanInfo(TEXT("gamebase://getBanInfo"));
}

namespace GamebaseCommunicatorAsyncSender
{
    void Login(const FGamebaseCommunicatorSendData& SendData, FGamebaseCommunicatorManager* Communicator, const FGamebaseAuthTokenDelegate& Callback)
    {
        Communicator->RegisterAsyncEvent(SendData, [=](const FGamebaseCommunicatorReceiveDataRef& ReceiveData)
        {
            const auto Error = GetGamebaseError(ReceiveData);
            if (ReceiveData->JsonData.IsEmpty())
            {
                Callback.ExecuteIfBound(nullptr, Error.Get());
                return;
            }
            
            const TSharedPtr<FGamebaseAuthToken, ESPMode::ThreadSafe> AuthToken = MakeShared<FGamebaseAuthToken, ESPMode::ThreadSafe>();
            AuthToken->FromJson(ReceiveData->JsonData);
            
            Callback.ExecuteIfBound(AuthToken.Get(), Error.Get());
        });
    }

    void AddMapping(const FGamebaseCommunicatorSendData& SendData, FGamebaseCommunicatorManager* Communicator, const FGamebaseAuthTokenDelegate& Callback)
    {
        Communicator->RegisterAsyncEvent(SendData, [=](const FGamebaseCommunicatorReceiveDataRef& ReceiveData)
        {
            const auto Error = GetGamebaseError(ReceiveData);
            if (ReceiveData->JsonData.IsEmpty())
            {
                Callback.ExecuteIfBound(nullptr, Error.Get());
                return;
            }
            
            const TSharedPtr<FGamebaseAuthToken, ESPMode::ThreadSafe> AuthToken = MakeShared<FGamebaseAuthToken, ESPMode::ThreadSafe>();
            AuthToken->FromJson(ReceiveData->JsonData);
            
            Callback.ExecuteIfBound(AuthToken.Get(), Error.Get());
        });
    }

    void TransferAccount(const FGamebaseCommunicatorSendData& SendData, FGamebaseCommunicatorManager* Communicator, const FGamebaseTransferAccountDelegate& Callback)
    {
        Communicator->RegisterAsyncEvent(SendData, [=](const FGamebaseCommunicatorReceiveDataRef& ReceiveData)
        {
            const auto Error = GetGamebaseError(ReceiveData);
            if (ReceiveData->JsonData.IsEmpty())
            {
                Callback.ExecuteIfBound(nullptr, Error.Get());
                return;
            }
            
            const FGamebaseTransferAccountInfoPtr TransferAccountInfo = MakeShared<FGamebaseTransferAccountInfo, ESPMode::ThreadSafe>();
            TransferAccountInfo->FromJson(ReceiveData->JsonData);
            
            Callback.ExecuteIfBound(TransferAccountInfo.Get(), Error.Get());
        });
    }

    void RequestTemporaryWithdrawal(const FGamebaseCommunicatorSendData& SendData, FGamebaseCommunicatorManager* Communicator, const FGamebaseTemporaryWithdrawalDelegate& Callback)
    {
        Communicator->RegisterAsyncEvent(SendData, [=](const FGamebaseCommunicatorReceiveDataRef& ReceiveData)
        {
            const auto Error = GetGamebaseError(ReceiveData);
            if (ReceiveData->JsonData.IsEmpty())
            {
                Callback.ExecuteIfBound(nullptr, Error.Get());
                return;
            }
            
            const TSharedPtr<FGamebaseTemporaryWithdrawalInfo, ESPMode::ThreadSafe> TransferAccountInfo = MakeShared<FGamebaseTemporaryWithdrawalInfo, ESPMode::ThreadSafe>();
            TransferAccountInfo->FromJson(ReceiveData->JsonData);
            
            Callback.ExecuteIfBound(TransferAccountInfo.Get(), Error.Get());
        });
    }
}

TSharedPtr<FJsonObject> ConvertTMapToJsonObject(const TMap<FName, FVariant>& Map)
{
    TSharedPtr<FJsonObject> JsonObject = MakeShared<FJsonObject>();

    for (const auto& Pair : Map)
    {
        FString Key = Pair.Key.ToString();
        const FVariant& Value = Pair.Value;

        switch (Value.GetType())
        {
            case EVariantTypes::Int32:
                JsonObject->SetNumberField(Key, Value.GetValue<int32>());
                break;
            case EVariantTypes::Float:
                JsonObject->SetNumberField(Key, Value.GetValue<float>());
                break;
            case EVariantTypes::Bool:
                JsonObject->SetBoolField(Key, Value.GetValue<bool>());
                break;
            case EVariantTypes::String:
                JsonObject->SetStringField(Key, Value.GetValue<FString>());
                break;
            case EVariantTypes::Name:
                JsonObject->SetStringField(Key, Value.GetValue<FName>().ToString());
                break;
            default:
                GAMEBASE_LOG_GLOBAL_WARNING("Unsupported FVariant type for key: %s", *Key);
                break;
        }
    }

    return JsonObject;
}

FString ConvertJsonObjectToString(const TSharedPtr<FJsonObject>& JsonObject)
{
    FString JsonString;
    const TSharedRef<TJsonWriter<>> Writer = TJsonWriterFactory<>::Create(&JsonString);

    if (FJsonSerializer::Serialize(JsonObject.ToSharedRef(), Writer))
    {
        JsonString.ReplaceInline(TEXT("\t"), TEXT(""));
        JsonString.ReplaceInline(TEXT("\n"), TEXT(""));
        return JsonString;
    }

    return TEXT("{}");
}

FString ConvertTMapToJsonString(const TMap<FName, FVariant>& Map)
{
    const TSharedPtr<FJsonObject> JsonObject = ConvertTMapToJsonObject(Map);
    return ConvertJsonObjectToString(JsonObject);
}

TSharedPtr<FJsonObject> ConvertFGamebaseVariantMapToJsonObject(const FGamebaseVariantMap& Map)
{
    TSharedPtr<FJsonObject> JsonObject = MakeShared<FJsonObject>();
    for (const auto& Pair : Map.GetMap())
    {
        FString Key = Pair.Key.ToString();
        const FGamebaseVariant& Value = Pair.Value;
        switch (Value.GetType())
        {
            case EGamebaseVariantType::Int32:
                JsonObject->SetNumberField(Key, Value.GetValue<int32>());
                break;
            case EGamebaseVariantType::Float:
                JsonObject->SetNumberField(Key, Value.GetValue<float>());
                break;
            case EGamebaseVariantType::Bool:
                JsonObject->SetBoolField(Key, Value.GetValue<bool>());
                break;
            case EGamebaseVariantType::String:
                JsonObject->SetStringField(Key, Value.GetValue<FString>());
                break;
            case EGamebaseVariantType::Name:
                JsonObject->SetStringField(Key, Value.GetValue<FName>().ToString());
                break;
            default:
                GAMEBASE_LOG_GLOBAL_WARNING("Unsupported FGamebaseVariant type for key: %s", *Key);
                break;
        }
    }
    return JsonObject;
}

FString ConvertFGamebaseVariantMapToJsonString(const FGamebaseVariantMap& Map)
{
    const TSharedPtr<FJsonObject> JsonObject = ConvertFGamebaseVariantMapToJsonObject(Map);
    return ConvertJsonObjectToString(JsonObject);
}

FGamebaseBridgeAuth::FGamebaseBridgeAuth(const TSharedPtr<FGamebaseCommunicatorManager>& Communicator, const FName& ClassName)
    : FGamebaseBridgeBase(Communicator, ClassName)
{
    if (UGamebaseBridgeSubsystem* Subsystem = GetSubsystem())
    {
        LoadedAdapters.Initialize(Subsystem);

        LoadedAdapters.ForEach([](UGamebaseBridgeAuthAdapter* Adapter)
        {
            if (Adapter)
            {
                Adapter->Initialize();
            }
        });

        GAMEBASE_LOG_GLOBAL_WARNING("Number of loaded adapters: %d", LoadedAdapters.GetAdapterArray<UGamebaseBridgeAuthAdapter>(UGamebaseBridgeAuthAdapter::StaticClass()).Num());
    }
    else
    {
        GAMEBASE_LOG_GLOBAL_WARNING("Failed to initialize FGamebaseBridgeAuth: Subsystem is invalid.");
    }
}

void FGamebaseBridgeAuth::Login(const FString& ProviderName, const FGamebaseAuthTokenDelegate& Callback)
{
    if (const auto Adapter = LoadedAdapters.GetAdapter(ProviderName))
    {
        Adapter->Login([this, Callback](const FGamebaseAuthCredentialsResult& Result)
        {
            if (Result.IsError())
            {
                Callback.ExecuteIfBound(nullptr, &Result.GetErrorValue());
                return;
            }
            
            Login(Result.GetOkValue(), Callback);
        });
        
        return;
    }
    
    FAuthProviderInfo LoginData;
    LoginData.ProviderName = ProviderName;

    FGamebaseCommunicatorSendData SendData(GamebaseScheme::Login);
    SendData.JsonData = LoginData.ToJson(false);

    GamebaseCommunicatorAsyncSender::Login(SendData, Communicator.Get(), Callback);
}

void FGamebaseBridgeAuth::Login(const FString& ProviderName, const FGamebaseVariantMap& AdditionalInfo, const FGamebaseAuthTokenDelegate& Callback)
{
    if (const auto Adapter = LoadedAdapters.GetAdapter(ProviderName))
    {
        Adapter->Login(AdditionalInfo, [this, Callback](const FGamebaseAuthCredentialsResult& Result)
        {
            if (Result.IsError())
            {
                Callback.ExecuteIfBound(nullptr, &Result.GetErrorValue());
                return;
            }
            Login(Result.GetOkValue(), Callback);
        });
        return;
    }
    const TSharedPtr<FJsonObject> JsonObject = MakeShared<FJsonObject>();
    JsonObject->SetStringField(TEXT("providerName"), ProviderName);
    JsonObject->SetObjectField(TEXT("additionalInfo"), ConvertFGamebaseVariantMapToJsonObject(AdditionalInfo));
    FGamebaseCommunicatorSendData SendData(GamebaseScheme::LoginWithAdditionalInfo);
    SendData.JsonData = ConvertJsonObjectToString(JsonObject);
    GamebaseCommunicatorAsyncSender::Login(SendData, Communicator.Get(), Callback);
}

void FGamebaseBridgeAuth::Login(const FGamebaseVariantMap& CredentialInfo, const FGamebaseAuthTokenDelegate& Callback)
{
    FGamebaseCommunicatorSendData SendData(GamebaseScheme::LoginWithCredentialInfo);
    SendData.JsonData = ConvertFGamebaseVariantMapToJsonString(CredentialInfo);
    GamebaseCommunicatorAsyncSender::Login(SendData, Communicator.Get(), Callback);
}

void FGamebaseBridgeAuth::LoginForLastLoggedInProvider(const FGamebaseAuthTokenDelegate& Callback)
{
    FGamebaseCommunicatorSendData SendData(GamebaseScheme::LoginForLastLoggedInProvider);
    
    GamebaseCommunicatorAsyncSender::Login(SendData, Communicator.Get(), Callback);
}

void FGamebaseBridgeAuth::LoginForLastLoggedInProvider(const FGamebaseVariantMap& AdditionalInfo, const FGamebaseAuthTokenDelegate& Callback)
{
    const TSharedPtr<FJsonObject> JsonObject = MakeShared<FJsonObject>();
    JsonObject->SetObjectField(TEXT("additionalInfo"), ConvertFGamebaseVariantMapToJsonObject(AdditionalInfo));
    FGamebaseCommunicatorSendData SendData(GamebaseScheme::LoginForLastLoggedInProviderWithAdditionalInfo);
    SendData.JsonData = ConvertJsonObjectToString(JsonObject);
    GamebaseCommunicatorAsyncSender::Login(SendData, Communicator.Get(), Callback);
}

void FGamebaseBridgeAuth::AddMapping(const FString& ProviderName, const FGamebaseAuthTokenDelegate& Callback)
{
    if (const auto Adapter = LoadedAdapters.GetAdapter(ProviderName))
    {
        Adapter->Login([this, Callback](const FGamebaseAuthCredentialsResult& Result)
        {
            if (Result.IsError())
            {
                Callback.ExecuteIfBound(nullptr, &Result.GetErrorValue());
                return;
            }
            
            AddMapping(Result.GetOkValue(), Callback);
        });
        
        return;
    }
    
    FAuthProviderInfo MappingData;
    MappingData.ProviderName = ProviderName;

    FGamebaseCommunicatorSendData SendData(GamebaseScheme::AddMapping);
    SendData.JsonData = MappingData.ToJson(false);

    GamebaseCommunicatorAsyncSender::AddMapping(SendData, Communicator.Get(), Callback);
}

void FGamebaseBridgeAuth::AddMapping(const FString& ProviderName, const FGamebaseVariantMap& AdditionalInfo, const FGamebaseAuthTokenDelegate& Callback)
{
    if (const auto Adapter = LoadedAdapters.GetAdapter(ProviderName))
    {
        Adapter->Login(AdditionalInfo, [this, Callback](const FGamebaseAuthCredentialsResult& Result)
        {
            if (Result.IsError())
            {
                Callback.ExecuteIfBound(nullptr, &Result.GetErrorValue());
                return;
            }
            AddMapping(Result.GetOkValue(), Callback);
        });
        return;
    }
    const TSharedPtr<FJsonObject> JsonObject = MakeShared<FJsonObject>();
    JsonObject->SetStringField(TEXT("providerName"), ProviderName);
    JsonObject->SetObjectField(TEXT("additionalInfo"), ConvertFGamebaseVariantMapToJsonObject(AdditionalInfo));
    FGamebaseCommunicatorSendData SendData(GamebaseScheme::AddMappingWithAdditionalInfo);
    SendData.JsonData = ConvertJsonObjectToString(JsonObject);
    GamebaseCommunicatorAsyncSender::AddMapping(SendData, Communicator.Get(), Callback);
}

void FGamebaseBridgeAuth::AddMapping(const FGamebaseVariantMap& CredentialInfo, const FGamebaseAuthTokenDelegate& Callback)
{
    FGamebaseCommunicatorSendData SendData(GamebaseScheme::AddMappingWithCredentialInfo);
    SendData.JsonData = ConvertFGamebaseVariantMapToJsonString(CredentialInfo);
    GamebaseCommunicatorAsyncSender::AddMapping(SendData, Communicator.Get(), Callback);
}

void FGamebaseBridgeAuth::ChangeLogin(const FGamebaseForcingMappingTicket& ForcingMappingTicket, const FGamebaseAuthTokenDelegate& Callback)
{
    FGamebaseCommunicatorSendData SendData(GamebaseScheme::ChangeLogin);
    SendData.JsonData = ForcingMappingTicket.ToJson();

    GamebaseCommunicatorAsyncSender::Login(SendData, Communicator.Get(), Callback);
}

void FGamebaseBridgeAuth::AddMappingForcibly(const FGamebaseForcingMappingTicket& ForcingMappingTicket, const FGamebaseAuthTokenDelegate& Callback)
{
    FGamebaseCommunicatorSendData SendData(GamebaseScheme::AddMappingForciblyWithForcingMappingTicket);
    SendData.JsonData = ForcingMappingTicket.ToJson();

    GamebaseCommunicatorAsyncSender::Login(SendData, Communicator.Get(), Callback);
}

void FGamebaseBridgeAuth::AddMappingForcibly(const FString& ProviderName, const FString& ForcingMappingKey, const FGamebaseAuthTokenDelegate& Callback)
{
    FAuthAddMappingForcibly MappingData;
    MappingData.ProviderName = ProviderName;
    MappingData.ForcingMappingKey = ForcingMappingKey;
    
    FGamebaseCommunicatorSendData SendData(GamebaseScheme::AddMappingForcibly);
    SendData.JsonData = MappingData.ToJson(false);
    
    GamebaseCommunicatorAsyncSender::AddMapping(SendData, Communicator.Get(), Callback);
}

void FGamebaseBridgeAuth::AddMappingForcibly(const FString& ProviderName, const FString& ForcingMappingKey, const FGamebaseVariantMap& AdditionalInfo, const FGamebaseAuthTokenDelegate& Callback)
{
    const TSharedPtr<FJsonObject> JsonObject = MakeShared<FJsonObject>();
    JsonObject->SetStringField(TEXT("providerName"), ProviderName);
    JsonObject->SetStringField(TEXT("forcingMappingKey"), ForcingMappingKey);
    JsonObject->SetObjectField(TEXT("additionalInfo"), ConvertFGamebaseVariantMapToJsonObject(AdditionalInfo));
    FGamebaseCommunicatorSendData SendData(GamebaseScheme::AddMappingForciblyWithAdditionalInfo);
    SendData.JsonData = ConvertJsonObjectToString(JsonObject);
    GamebaseCommunicatorAsyncSender::AddMapping(SendData, Communicator.Get(), Callback);
}

void FGamebaseBridgeAuth::AddMappingForcibly(const FGamebaseVariantMap& CredentialInfo, const FString& ForcingMappingKey, const FGamebaseAuthTokenDelegate& Callback)
{
    const TSharedPtr<FJsonObject> JsonObject = MakeShared<FJsonObject>();
    JsonObject->SetStringField(TEXT("forcingMappingKey"), ForcingMappingKey);
    JsonObject->SetObjectField(TEXT("additionalInfo"), ConvertFGamebaseVariantMapToJsonObject(CredentialInfo));
    FGamebaseCommunicatorSendData SendData(GamebaseScheme::AddMappingForciblyWithCredentialInfo);
    SendData.JsonData = ConvertJsonObjectToString(JsonObject);
    GamebaseCommunicatorAsyncSender::AddMapping(SendData, Communicator.Get(), Callback);
}

void FGamebaseBridgeAuth::RemoveMapping(const FString& ProviderName, const FGamebaseErrorDelegate& Callback)
{
    FAuthProviderInfo MappingData;
    MappingData.ProviderName = ProviderName;

    FGamebaseCommunicatorSendData SendData(GamebaseScheme::RemoveMapping);
    SendData.JsonData = MappingData.ToJson(false);
    
    GamebaseCommunicatorAsyncSender::Default(SendData, Communicator.Get(), Callback);
}

void FGamebaseBridgeAuth::Logout(const FGamebaseErrorDelegate& Callback)
{
    FGamebaseCommunicatorSendData SendData(GamebaseScheme::Logout);

    GamebaseCommunicatorAsyncSender::Default(SendData, Communicator.Get(), Callback);
}

void FGamebaseBridgeAuth::Withdraw(const FGamebaseErrorDelegate& Callback)
{
    FGamebaseCommunicatorSendData SendData(GamebaseScheme::Withdraw);

    GamebaseCommunicatorAsyncSender::Default(SendData, Communicator.Get(), Callback);
}

void FGamebaseBridgeAuth::WithdrawImmediately(const FGamebaseErrorDelegate& Callback)
{
    FGamebaseCommunicatorSendData SendData(GamebaseScheme::WithdrawImmediately);

    GamebaseCommunicatorAsyncSender::Default(SendData, Communicator.Get(), Callback);
}

void FGamebaseBridgeAuth::RequestTemporaryWithdrawal(const FGamebaseTemporaryWithdrawalDelegate& Callback)
{
    FGamebaseCommunicatorSendData SendData(GamebaseScheme::RequestTemporaryWithdrawal);

    GamebaseCommunicatorAsyncSender::RequestTemporaryWithdrawal(SendData, Communicator.Get(), Callback);
}

void FGamebaseBridgeAuth::CancelTemporaryWithdrawal(const FGamebaseErrorDelegate& Callback)
{
    FGamebaseCommunicatorSendData SendData(GamebaseScheme::CancelTemporaryWithdrawal);

    GamebaseCommunicatorAsyncSender::Default(SendData, Communicator.Get(), Callback);
}

void FGamebaseBridgeAuth::IssueTransferAccount(const FGamebaseTransferAccountDelegate& Callback)
{
    FGamebaseCommunicatorSendData SendData(GamebaseScheme::IssueTransferAccount);

    GamebaseCommunicatorAsyncSender::TransferAccount(SendData, Communicator.Get(), Callback);
}

void FGamebaseBridgeAuth::QueryTransferAccount(const FGamebaseTransferAccountDelegate& Callback)
{
    FGamebaseCommunicatorSendData SendData(GamebaseScheme::QueryTransferAccount);

    GamebaseCommunicatorAsyncSender::TransferAccount(SendData, Communicator.Get(), Callback);
}

void FGamebaseBridgeAuth::RenewTransferAccount(const FGamebaseTransferAccountRenewConfiguration& Configuration, const FGamebaseTransferAccountDelegate& Callback)
{
    FGamebaseCommunicatorSendData SendData(GamebaseScheme::RenewTransferAccount);
    SendData.JsonData = Configuration.ToJson();
    
    GamebaseCommunicatorAsyncSender::TransferAccount(SendData, Communicator.Get(), Callback);
}

void FGamebaseBridgeAuth::TransferAccountWithIdPLogin(const FString& AccountId, const FString& AccountPassword, const FGamebaseAuthTokenDelegate& Callback)
{
    FAuthTransferAccount AccountData;
    AccountData.AccountId = AccountId;
    AccountData.AccountPassword = AccountPassword;
    
    FGamebaseCommunicatorSendData SendData(GamebaseScheme::TransferAccountWithIdPLogin);
    SendData.JsonData = AccountData.ToJson(false);

    GamebaseCommunicatorAsyncSender::Login(SendData, Communicator.Get(), Callback);
}

TArray<FString> FGamebaseBridgeAuth::GetAuthMappingList()
{
    FGamebaseCommunicatorSendData SendData(GamebaseScheme::GetAuthMappingList);

    TArray<FString> MappingList;

    const FString GetJsonString = Communicator->GetSync(SendData.ToJson(false));
    if (GetJsonString.IsEmpty() == false)
    {
        const auto JsonReader = TJsonReaderFactory<>::Create(GetJsonString);
        TArray<TSharedPtr<FJsonValue>> JsonValues;

        if (FJsonSerializer::Deserialize(JsonReader, JsonValues) == true)
        {
            for (const auto& JsonValue : JsonValues)
            {
                MappingList.Add(JsonValue->AsString());
            }
        }
    }

    return MappingList;
}

FString FGamebaseBridgeAuth::GetAuthProviderUserID(const FString& ProviderName)
{
    FGamebaseCommunicatorSendData SendData(GamebaseScheme::GetAuthProviderUserID);
    SendData.JsonData = ProviderName;

    return Communicator->GetSync(SendData.ToJson(false));
}

FString FGamebaseBridgeAuth::GetAuthProviderAccessToken(const FString& ProviderName)
{
    FGamebaseCommunicatorSendData SendData(GamebaseScheme::GetAuthProviderAccessToken);
    SendData.JsonData = ProviderName;

    return Communicator->GetSync(SendData.ToJson(false));
}

const FGamebaseAuthProviderProfilePtr FGamebaseBridgeAuth::GetAuthProviderProfile(const FString& ProviderName)
{
    FGamebaseCommunicatorSendData SendData(GamebaseScheme::GetAuthProviderProfile);
    SendData.JsonData = ProviderName;

    const FString GetJsonString = Communicator->GetSync(SendData.ToJson(false));
    if (GetJsonString.IsEmpty())
    {
        return nullptr;
    }

    UGamebaseJsonObject* Information = NewObject<UGamebaseJsonObject>();
    if (Information->DecodeJson(GetJsonString) == false)
    {
        return nullptr;
    }

    const FGamebaseAuthProviderProfilePtr Profile = MakeShared<FGamebaseAuthProviderProfile, ESPMode::ThreadSafe>();
    Profile->Information = Information->GetObjectField("information");

    return Profile;
}

const FGamebaseBanInfoPtr FGamebaseBridgeAuth::GetBanInfo()
{
    FGamebaseCommunicatorSendData SendData(GamebaseScheme::GetBanInfo);

    const FString GetJsonString = Communicator->GetSync(SendData.ToJson(false));
    if (GetJsonString.IsEmpty())
    {
        return nullptr;
    }
    
    const FGamebaseBanInfoPtr BanInfo = MakeShared<FGamebaseBanInfo, ESPMode::ThreadSafe>();
    BanInfo->FromJson(GetJsonString);
    
    return BanInfo;
}