﻿#include "GamebaseCommunicatorManager.h"
#include "GamebaseCommunicator.h"
#include "GamebasePlatformDefines.h"
#include "Async/TaskGraphInterfaces.h"

FGamebaseCommunicatorManager::FGamebaseCommunicatorManager()
    : Communicator(MakeUnique<FGamebaseCommunicator>())
{
}

FGamebaseCommunicatorManager::~FGamebaseCommunicatorManager()
{
}

void FGamebaseCommunicatorManager::Initialize(const FName& ClassName)
{
    Communicator->Initialize(ClassName);
}

void FGamebaseCommunicatorManager::RegisterEvent(const FName& Scheme, const FCommunicatorCallback& Callback)
{
    FScopeLock Lock(&CriticalSection);
    SchemeCallbacks.FindOrAdd(Scheme, Callback);
}

void FGamebaseCommunicatorManager::RegisterAsyncEvent(FGamebaseCommunicatorSendData Data, FCommunicatorCallback Callback)
{   
    if (Callback != nullptr)
    {
        static FThreadSafeCounter HandleCounter;
        Data.Handle = HandleCounter.Increment();
        RegisterHandleCallback(Data.Handle, Callback);
    }

    const FString JsonString = Data.ToJson(false);
    Communicator->GetAsync(JsonString);
}

void FGamebaseCommunicatorManager::ReceiveEvent(FGamebaseCommunicatorReceiveDataPtr Result)
{
    check(Result.IsValid());

    FFunctionGraphTask::CreateAndDispatchWhenReady([this, Result]()
    {
        FScopeLock Lock(&CriticalSection);
        
        if (const FCommunicatorCallback* SchemeCallback = SchemeCallbacks.Find(FName(Result->Scheme)))
        {
            (*SchemeCallback)(Result.ToSharedRef());
        }
        
        if (const FCommunicatorCallback* HandleCallback = HandleCallbacks.Find(Result->Handle))
        {
            (*HandleCallback)(Result.ToSharedRef());
            UnregisterHandleCallback(Result->Handle);
        }
    }
    , TStatId(), nullptr, ENamedThreads::GameThread);
}

void FGamebaseCommunicatorManager::GetAsyncWithoutResponse(const FString& JsonString)
{
    Communicator->GetAsync(JsonString);
}

FString FGamebaseCommunicatorManager::GetSync(const FString& JsonString)
{
    return Communicator->GetSync(JsonString);
}

void FGamebaseCommunicatorManager::RegisterHandleCallback(const int32 Handle, const FCommunicatorCallback& Callback)
{
    FScopeLock Lock(&CriticalSection);
    HandleCallbacks.FindOrAdd(Handle, Callback);
}

void FGamebaseCommunicatorManager::UnregisterHandleCallback(const int32 Handle)
{
    FScopeLock Lock(&CriticalSection);
    HandleCallbacks.Remove(Handle);
}
