#pragma once

#include "GamebaseCommunicatorTypes.h"
#include "Types/GamebaseDataContainer.h"
#include "Types/GamebaseError.h"
#include "GamebaseCommunicationRequestTypes.h"

class IGamebaseCommunicator;
struct FGamebaseCommunicatorSendData;

namespace GamebaseCommunicatorAsyncSender
{
    static void Default(const FGamebaseCommunicatorSendData& SendData, FGamebaseCommunicatorManager* Communicator, const FGamebaseErrorDelegate& Callback)
    {
        Communicator->RegisterAsyncEvent(SendData, [=](const FGamebaseCommunicatorReceiveDataRef& ReceiveData)
        {
            Callback.ExecuteIfBound(GetGamebaseError(ReceiveData).Get());
        });
    }

    static void UseScheme(const FGamebaseCommunicatorSendData& SendData, FGamebaseCommunicatorManager* Communicator, const FGamebaseErrorDelegate& Callback, const FGamebaseExtraDataDelegate& SchemeRemoveCallback)
    {
        Communicator->RegisterAsyncEvent(SendData, [=](const FGamebaseCommunicatorReceiveDataRef& ReceiveData)
        {
            Callback.ExecuteIfBound(GetGamebaseError(ReceiveData).Get());
            SchemeRemoveCallback.ExecuteIfBound(ReceiveData->ExtraData);
        });
    }
    
    static void DataContainer(const FGamebaseCommunicatorSendData& SendData, FGamebaseCommunicatorManager* Communicator, const FGamebaseDataContainerDelegate& Callback)
    {
        Communicator->RegisterAsyncEvent(SendData, [=](const FGamebaseCommunicatorReceiveDataRef& ReceiveData)
        {
            const auto Error = GetGamebaseError(ReceiveData);
            if (ReceiveData->JsonData.IsEmpty())
            {
                Callback.ExecuteIfBound(nullptr, Error.Get());
                return;
            }

            const TSharedPtr<FGamebaseDataContainer, ESPMode::ThreadSafe> DataContainer = MakeShared<FGamebaseDataContainer, ESPMode::ThreadSafe>();
            DataContainer->FromJson(ReceiveData->JsonData);
            
            Callback.ExecuteIfBound(DataContainer.Get(), Error.Get());
        });
    }
}