﻿using System;
using System.Collections.Generic;
using System.IO;
using NHNCloud.Gamebase;
using UnrealBuildTool;

#if UE_5_0_OR_LATER
using EpicGames.Core;
#else
using Tools.DotNETCommon;
#endif

public class GamebaseBridge : ModuleRules
{
    private const string AndroidSettingSection = "/Script/GamebaseEditor.GamebaseAndroidPlatformSettings";
    private const string IosSettingSection = "/Script/GamebaseEditor.GamebaseIOSPlatformSettings";
    
    public GamebaseBridge(ReadOnlyTargetRules Target) : base(Target)
    {
        PCHUsage = PCHUsageMode.UseExplicitOrSharedPCHs;

        PrivateIncludePaths.AddRange(
            new[] {
                "GamebaseBridge/Private/Communicator",
            }
        );

        PrivateDependencyModuleNames.AddRange(
            new[]
            {
                "Core",
                "CoreUObject",
                "Engine",
                "Json",
                "JsonUtilities",
                "Gamebase",
                "GamebaseCore",
            }
        );
        
        if (Target.Platform == UnrealTargetPlatform.Android)
        {
            SetupAndroid();
        }
        else if (Target.Platform == UnrealTargetPlatform.IOS)
        {
            SetupIOS(); 
        }
        else
        {
            Console.WriteLine("[GamebaseBridge] {0} is not supported.", Target.Platform);
            //Logger.LogInfo("{0} is not supported.", Target.Platform);
        }
    }

    private void SetupAndroid()
    {
        PrivateIncludePaths.Add("GamebaseBridge/Private/Android");
        var pluginPath = Utils.MakePathRelativeTo(ModuleDirectory, Target.RelativeEnginePath);
        AdditionalPropertiesForReceipt.Add("AndroidPlugin", Path.Combine(pluginPath, "Gamebase_Android_UPL.xml"));
    }

    private void SetupIOS()
    {
        PrivateIncludePaths.Add("GamebaseBridge/Private/iOS");
        
        var useFrameworks = new List<string>();
        useFrameworks.AddRange(
            new[] {
                "GameKit",
                "AdSupport",
                "CoreTelephony",
                "StoreKit",
                "ImageIO",
                "MobileCoreServices",
        });
            
        bool bEnableGoogle = false;
        bool bEnableFacebook = false;
        bool bEnableAppleId = false;
        bool bEnableGamecenter = false;
        bool bEnableTwitter = false;
        bool bEnableLine = false;;
        bool bEnableNaver = false;
        bool bEnableWeibo = false;
        bool bEnableSteam = false;
        bool bEnableAppStore = false;
        bool bEnableAPNS = false;
        bool bEnableSwift = true;
        
        ConfigHierarchy config = ConfigCache.ReadHierarchy(ConfigHierarchyType.Engine, DirectoryReference.FromFile(Target.ProjectFile), UnrealTargetPlatform.IOS);
        config.GetBool(IosSettingSection, "bEnableGoogle", out bEnableGoogle);
        config.GetBool(IosSettingSection, "bEnableFacebook", out bEnableFacebook);
        config.GetBool(IosSettingSection, "bEnableAppleId", out bEnableAppleId);
        config.GetBool(IosSettingSection, "bEnableGamecenter", out bEnableGamecenter);
        config.GetBool(IosSettingSection, "bEnableTwitter", out bEnableTwitter);
        config.GetBool(IosSettingSection, "bEnableLine", out bEnableLine);
        config.GetBool(IosSettingSection, "bEnableNaver", out bEnableNaver);
        config.GetBool(IosSettingSection, "bEnableWeibo", out bEnableWeibo);
        config.GetBool(IosSettingSection, "bEnableSteam", out bEnableSteam);
        config.GetBool(IosSettingSection, "bEnableAppStore", out bEnableAppStore);
        config.GetBool(IosSettingSection, "bEnableAPNS", out bEnableAPNS);
        
        if (bEnableSwift)
        {
            string xcodePath;
            config.GetString(IosSettingSection, "xcodePath", out xcodePath);

            if (string.IsNullOrEmpty(xcodePath))
            {
                xcodePath = "/Applications/Xcode.app";
            }

            PublicSystemLibraryPaths.Add(xcodePath + "/Contents/Developer/Platforms/iPhoneOS.platform/Developer/SDKs/iPhoneOS.sdk/usr/lib/swift");
            PublicSystemLibraryPaths.Add(xcodePath + "/Contents/Developer/Toolchains/XcodeDefault.xctoolchain/usr/lib/swift/iphoneos");
            PublicSystemLibraryPaths.Add(xcodePath + "/Contents/Developer/Toolchains/XcodeDefault.xctoolchain/usr/lib/swift-5.0/iphoneos");
        }

        string iOSPluginPath = Path.GetFullPath(Path.Combine(ModuleDirectory, "ThirdParty/IOS"));

        PublicAdditionalFrameworks.Add(
            new Framework(
                "GamebasePlugin",
                Path.Combine(iOSPluginPath, "GamebasePlugin.embeddedframework.zip")
            )
        );
        
        PublicAdditionalFrameworks.Add(
            new Framework(
                "Gamebase",
                Path.Combine(iOSPluginPath, "Gamebase.embeddedframework.zip"),
                "Gamebase.bundle"
            )
        );
        
        PublicAdditionalFrameworks.Add(
            new Framework(
                "NHNCloudIAP",
                Path.Combine(iOSPluginPath, "NHNCloudIAP.embeddedframework.zip")
            )
        );

        PublicAdditionalFrameworks.Add(
            new Framework(
                "NHNCloudPush",
                Path.Combine(iOSPluginPath, "NHNCloudPush.embeddedframework.zip")
            )
        );
        
        if (bEnableGoogle)
        {
            PublicAdditionalFrameworks.Add(
                new Framework(
                    "GamebaseAuthGoogleAdapter",
                    Path.Combine(iOSPluginPath, "GamebaseAuthGoogleAdapter.embeddedframework.zip")
                )
            );

            PublicAdditionalFrameworks.Add(
                new Framework(
                    "GoogleSignIn",
                    Path.Combine(iOSPluginPath, "GoogleSignIn.embeddedframework.zip"),
                    "GoogleSignIn.bundle"
                )
            );

            PublicAdditionalFrameworks.Add(
                new Framework(
                    "GTMAppAuth",
                    Path.Combine(iOSPluginPath, "GTMAppAuth.embeddedframework.zip")
                )
            );

            PublicAdditionalFrameworks.Add(
                new Framework(
                    "GTMSessionFetcher",
                    Path.Combine(iOSPluginPath, "GTMSessionFetcher.embeddedframework.zip")
                )
            );

            PublicAdditionalFrameworks.Add(
                new Framework(
                    "AppAuth",
                    Path.Combine(iOSPluginPath, "AppAuth.embeddedframework.zip")
                )
            );
        }
        
        if (bEnableFacebook)
        {
            PublicAdditionalFrameworks.Add(
                new Framework(
                    "GamebaseAuthFacebookAdapter",
                    Path.Combine(iOSPluginPath, "GamebaseAuthFacebookAdapter.embeddedframework.zip")
                )
            );
            
            PublicAdditionalFrameworks.Add(
                new Framework(
                    "FBAEMKit",
                    Path.Combine(iOSPluginPath, "FBAEMKit.embeddedframework.zip")
                )
            );

            PublicAdditionalFrameworks.Add(
                new Framework(
                    "FBSDKCoreKit",
                    Path.Combine(iOSPluginPath, "FBSDKCoreKit.embeddedframework.zip")
                )
            );
            
            PublicAdditionalFrameworks.Add(
                new Framework(
                    "FBSDKCoreKit_Basics",
                    Path.Combine(iOSPluginPath, "FBSDKCoreKit_Basics.embeddedframework.zip")
                )
            );
            
            PublicAdditionalFrameworks.Add(
                new Framework(
                    "FBSDKLoginKit",
                    Path.Combine(iOSPluginPath, "FBSDKLoginKit.embeddedframework.zip")
                )
            );
            
            useFrameworks.Add("Accelerate");
        }
        
        if (bEnableAppleId)
        {
            PublicAdditionalFrameworks.Add(
                new Framework(
                    "GamebaseAuthAppleidAdapter",
                    Path.Combine(iOSPluginPath, "GamebaseAuthAppleidAdapter.embeddedframework.zip")
                )
            );
        }
        
        if (bEnableGamecenter)
        {
            PublicAdditionalFrameworks.Add(
                new Framework(
                    "GamebaseAuthGamecenterAdapter",
                    Path.Combine(iOSPluginPath, "GamebaseAuthGamecenterAdapter.embeddedframework.zip")
                )
            );
        }
        
        if (bEnableLine)
        {
            PublicAdditionalFrameworks.Add(
                new Framework(
                    "GamebaseAuthLineAdapter",
                    Path.Combine(iOSPluginPath, "GamebaseAuthLineAdapter.embeddedframework.zip")
                )
            );
            
            PublicAdditionalFrameworks.Add(
                new Framework(
                    "LineSDK",
                    Path.Combine(iOSPluginPath, "LineSDK.embeddedframework.zip")
                )
            );
        }
        
        if (bEnableNaver)
        {
            PublicAdditionalFrameworks.Add(
                new Framework(
                    "GamebaseAuthNaverAdapter",
                    Path.Combine(iOSPluginPath, "GamebaseAuthNaverAdapter.embeddedframework.zip")
                )
            );
        
            PublicAdditionalFrameworks.Add(
                new Framework(
                    "NaverThirdPartyLogin",
                   Path.Combine(iOSPluginPath, "NaverThirdPartyLogin.embeddedframework.zip")
                )
            );
        }
        
        if (bEnableTwitter)
        {
            PublicAdditionalFrameworks.Add(
                new Framework(
                    "GamebaseAuthTwitterAdapter",
                    Path.Combine(iOSPluginPath, "GamebaseAuthTwitterAdapter.embeddedframework.zip")
                )
            );
        }
        
        if (bEnableWeibo)
        {
            PublicAdditionalFrameworks.Add(
                new Framework(
                    "GamebaseAuthWeiboAdapter",
                    Path.Combine(iOSPluginPath, "GamebaseAuthWeiboAdapter.embeddedframework.zip"),
                    "WeiboSDK.bundle"
                )
            );
            
            PublicAdditionalLibraries.Add(iOSPluginPath + "/libWeiboSDK.a");
        }
        
        if (bEnableSteam)
        {
            PublicAdditionalFrameworks.Add(
                new Framework(
                    "GamebaseAuthSteamAdapter",
                    Path.Combine(iOSPluginPath, "GamebaseAuthSteamAdapter.embeddedframework.zip")
                )
            );
        }
        
        if (bEnableAppStore)
        {
            PublicAdditionalFrameworks.Add(
                new Framework(
                    "GamebasePurchaseIAPAdapter",
                    Path.Combine(iOSPluginPath, "GamebasePurchaseIAPAdapter.embeddedframework.zip")
                )
            );
        }
        
        if (bEnableAPNS)
        {
            PublicAdditionalFrameworks.Add(
                new Framework(
                    "GamebasePushAdapter",
                    Path.Combine(iOSPluginPath, "GamebasePushAdapter.embeddedframework.zip")
                )
            );
        }
        
        PublicFrameworks.AddRange(useFrameworks);
        
        string PluginPath = Utils.MakePathRelativeTo(ModuleDirectory, Target.RelativeEnginePath);
        AdditionalPropertiesForReceipt.Add("IOSPlugin", Path.Combine(PluginPath, "Gamebase_iOS_UPL.xml"));
    }
}