#pragma once

#include "Dom/JsonObject.h"
#include "Dom/JsonValue.h"
#include "GamebaseJsonTypes.generated.h"

class UGamebaseJsonValue;

UCLASS(BlueprintType, Blueprintable)
class GAMEBASE_API UGamebaseJsonObject : public UObject
{
    GENERATED_UCLASS_BODY()

    /** Create new Json object, cannot be pure  */
    UFUNCTION(BlueprintCallable, meta = (DisplayName = "Construct Json Object", KeyWords = "create make", HidePin = "WorldContextObject", DefaultToSelf = "WorldContextObject"), Category = "Gamebase|Json")
    static UGamebaseJsonObject* ConstructJsonObject(UObject* WorldContextObject);

    /** Reset all internal data */
    UFUNCTION(BlueprintCallable, Category = "Gamebase|Json")
    void Reset();

    /** Get the root Json object */
    const TSharedPtr<FJsonObject>& GetRootObject() const;

    /** Set the root Json object */
    void SetRootObject(const TSharedPtr<FJsonObject>& JsonObject);


    //////////////////////////////////////////////////////////////////////////
    // Serialization

    /** Serialize Json to string (formatted with line breaks) */
    UFUNCTION(BlueprintCallable, Category = "Gamebase|Json")
    FString EncodeJson() const;

    /** Serialize Json to string (single string without line breaks) */
    UFUNCTION(BlueprintCallable, Category = "Gamebase|Json")
    FString EncodeJsonToSingleString() const;

    /** Construct Json object from string */
    UFUNCTION(BlueprintCallable, Category = "Gamebase|Json")
    bool DecodeJson(const FString& JsonString);


    //////////////////////////////////////////////////////////////////////////
    // FJsonObject API

    /** Returns a list of field names that exist in the object */
    UFUNCTION(BlueprintPure, Category = "Gamebase|Json")
    TArray<FString> GetFieldNames();

    /** Checks to see if the FieldName exists in the object */
    UFUNCTION(BlueprintCallable, Category = "Gamebase|Json")
    bool HasField(const FString& FieldName) const;

    /** Remove field named FieldName */
    UFUNCTION(BlueprintCallable, Category = "Gamebase|Json")
    void RemoveField(const FString& FieldName);

    /** Get the field named FieldName as a JsonValue */
    UFUNCTION(BlueprintCallable, Category = "Gamebase|Json")
    UGamebaseJsonValue* GetField(const FString& FieldName) const;

    /** Add a field named FieldName with a Value */
    UFUNCTION(BlueprintCallable, Category = "Gamebase|Json")
    void SetField(const FString& FieldName, UGamebaseJsonValue* JsonValue);

    /** Add a field named FieldName with a null value */
    UFUNCTION(BlueprintCallable, Category = "Gamebase | Json")
    void SetFieldNull(const FString& FieldName);

    /** Get the field named FieldName as a Json Array */
    UFUNCTION(BlueprintCallable, Category = "Gamebase|Json")
    TArray<UGamebaseJsonValue*> GetArrayField(const FString& FieldName);

    /** Set an ObjectField named FieldName and value of Json Array */
    UFUNCTION(BlueprintCallable, Category = "Gamebase|Json")
    void SetArrayField(const FString& FieldName, const TArray<UGamebaseJsonValue*>& InArray);

    /** Adds all of the fields from one json object to this one */
    UFUNCTION(BlueprintCallable, Category = "Gamebase|Json")
    void MergeJsonObject(UGamebaseJsonObject* InJsonObject, bool Overwrite);


    //////////////////////////////////////////////////////////////////////////
    // FJsonObject API Helpers (easy to use with simple Json objects)

    /** Get the field named FieldName as a number. Ensures that the field is present and is of type Json number.
     * Attn.!! float used instead of double to make the function blueprintable! */
    UFUNCTION(BlueprintCallable, Category = "Gamebase|Json")
    float GetNumberField(const FString& FieldName) const;

    /** Add a field named FieldName with Number as value
     * Attn.!! float used instead of double to make the function blueprintable! */
    UFUNCTION(BlueprintCallable, Category = "Gamebase|Json")
    void SetNumberField(const FString& FieldName, float Number);

    /** Get the field named FieldName as a string. */
    UFUNCTION(BlueprintCallable, Category = "Gamebase|Json")
    FString GetStringField(const FString& FieldName) const;

    /** Add a field named FieldName with value of StringValue */
    UFUNCTION(BlueprintCallable, Category = "Gamebase|Json")
    void SetStringField(const FString& FieldName, const FString& StringValue);

    /** Get the field named FieldName as a boolean. */
    UFUNCTION(BlueprintCallable, Category = "Gamebase|Json")
    bool GetBoolField(const FString& FieldName) const;

    /** Set a boolean field named FieldName and value of InValue */
    UFUNCTION(BlueprintCallable, Category = "Gamebase|Json")
    void SetBoolField(const FString& FieldName, bool InValue);

    /** Get the field named FieldName as a Json object. */
    UFUNCTION(BlueprintCallable, Category = "Gamebase|Json")
    UGamebaseJsonObject* GetObjectField(const FString& FieldName) const;

    /** Set an ObjectField named FieldName and value of JsonObject */
    UFUNCTION(BlueprintCallable, Category = "Gamebase|Json")
    void SetObjectField(const FString& FieldName, const UGamebaseJsonObject* JsonObject);
    

    //////////////////////////////////////////////////////////////////////////
    // Array fields helpers (uniform arrays)

    /** Get the field named FieldName as a Number Array. Use it only if you're sure that array is uniform!
     * Attn.!! float used instead of double to make the function blueprintable! */
    UFUNCTION(BlueprintCallable, Category = "Gamebase|Json")
    TArray<float> GetNumberArrayField(const FString& FieldName);

    /** Set an ObjectField named FieldName and value of Number Array
     * Attn.!! float used instead of double to make the function blueprintable! */
    UFUNCTION(BlueprintCallable, Category = "Gamebase|Json")
    void SetNumberArrayField(const FString& FieldName, const TArray<float>& NumberArray);

    /** Get the field named FieldName as a String Array. Use it only if you're sure that array is uniform! */
    UFUNCTION(BlueprintCallable, Category = "Gamebase|Json")
    TArray<FString> GetStringArrayField(const FString& FieldName);

    /** Set an ObjectField named FieldName and value of String Array */
    UFUNCTION(BlueprintCallable, Category = "Gamebase|Json")
    void SetStringArrayField(const FString& FieldName, const TArray<FString>& StringArray);

    /** Get the field named FieldName as a Bool Array. Use it only if you're sure that array is uniform! */
    UFUNCTION(BlueprintCallable, Category = "Gamebase|Json")
    TArray<bool> GetBoolArrayField(const FString& FieldName);

    /** Set an ObjectField named FieldName and value of Bool Array */
    UFUNCTION(BlueprintCallable, Category = "Gamebase|Json")
    void SetBoolArrayField(const FString& FieldName, const TArray<bool>& BoolArray);

    /** Get the field named FieldName as an Object Array. Use it only if you're sure that array is uniform! */
    UFUNCTION(BlueprintCallable, Category = "Gamebase|Json")
    TArray<UGamebaseJsonObject*> GetObjectArrayField(const FString& FieldName);

    /** Set an ObjectField named FieldName and value of Ob Array */
    UFUNCTION(BlueprintCallable, Category = "Gamebase|Json")
    void SetObjectArrayField(const FString& FieldName, const TArray<UGamebaseJsonObject*>& ObjectArray);


    //////////////////////////////////////////////////////////////////////////
    // Data

private:
    /** Internal JSON data */
    TSharedPtr<FJsonObject> JsonObj;
};


/**
 * Represents all the types a Json Value can be.
 */
UENUM(BlueprintType)
namespace EGamebaseJson
{
    enum Type
    {
        None,
        Null,
        String,
        Number,
        Boolean,
        Array,
        Object
    };
}


/**
 * Blueprintable FJsonValue wrapper
 */
UCLASS(BlueprintType, Blueprintable)
class UGamebaseJsonValue : public UObject
{
    GENERATED_UCLASS_BODY()

public:

    /** Create new Json Number value
     * Attn.!! float used instead of double to make the function blueprintable! */
    UFUNCTION(BlueprintPure, meta = (DisplayName = "Construct Json Number Value", HidePin = "WorldContextObject", DefaultToSelf = "WorldContextObject"), Category = "Gamebase|Json")
    static UGamebaseJsonValue* ConstructJsonValueNumber(UObject* WorldContextObject, float Number);

    /** Create new Json String value */
    UFUNCTION(BlueprintPure, meta = (DisplayName = "Construct Json String Value", HidePin = "WorldContextObject", DefaultToSelf = "WorldContextObject"), Category = "Gamebase|Json")
    static UGamebaseJsonValue* ConstructJsonValueString(UObject* WorldContextObject, const FString& StringValue);

    /** Create new Json Bool value */
    UFUNCTION(BlueprintPure, meta = (DisplayName = "Construct Json Bool Value", HidePin = "WorldContextObject", DefaultToSelf = "WorldContextObject"), Category = "Gamebase|Json")
    static UGamebaseJsonValue* ConstructJsonValueBool(UObject* WorldContextObject, bool InValue);

    /** Create new Json Array value */
    UFUNCTION(BlueprintPure, meta = (DisplayName = "Construct Json Array Value", HidePin = "WorldContextObject", DefaultToSelf = "WorldContextObject"), Category = "Gamebase|Json")
    static UGamebaseJsonValue* ConstructJsonValueArray(UObject* WorldContextObject, const TArray<UGamebaseJsonValue*>& InArray);

    /** Create new Json Object value */
    UFUNCTION(BlueprintPure, meta = (DisplayName = "Construct Json Object Value", HidePin = "WorldContextObject", DefaultToSelf = "WorldContextObject"), Category = "Gamebase|Json")
    static UGamebaseJsonValue* ConstructJsonValueObject(UGamebaseJsonObject *JsonObject, UObject* WorldContextObject);

    /** Create new Json value from FJsonValue (to be used from UGamebaseJsonObject) */
    static UGamebaseJsonValue* ConstructJsonValue(UObject* WorldContextObject, const TSharedPtr<FJsonValue>& InValue);

    /** Get the root Json value */
    TSharedPtr<FJsonValue>& GetRootValue();

    /** Set the root Json value */
    void SetRootValue(TSharedPtr<FJsonValue>& JsonValue);


    //////////////////////////////////////////////////////////////////////////
    // FJsonValue API

    /** Get type of Json value (Enum) */
    UFUNCTION(BlueprintCallable, Category = "Gamebase|Json")
    EGamebaseJson::Type GetType() const;

    /** Get type of Json value (String) */
    UFUNCTION(BlueprintCallable, Category = "Gamebase|Json")
    FString GetTypeString() const;

    /** Returns true if this value is a 'null' */
    UFUNCTION(BlueprintCallable, Category = "Gamebase|Json")
    bool IsNull() const;

    /** Returns this value as a double, throwing an error if this is not an Json Number
     * Attn.!! float used instead of double to make the function blueprintable! */
    UFUNCTION(BlueprintCallable, Category = "Gamebase|Json")
    float AsNumber() const;

    /** Returns this value as a number, throwing an error if this is not an Json String */
    UFUNCTION(BlueprintCallable, Category = "Gamebase|Json")
    FString AsString() const;

    /** Returns this value as a boolean, throwing an error if this is not an Json Bool */
    UFUNCTION(BlueprintCallable, Category = "Gamebase|Json")
    bool AsBool() const;

    /** Returns this value as an array, throwing an error if this is not an Json Array */
    UFUNCTION(BlueprintCallable, Category = "Gamebase|Json")
    TArray<UGamebaseJsonValue*> AsArray() const;

    /** Returns this value as an object, throwing an error if this is not an Json Object */
    UFUNCTION(BlueprintCallable, Category = "Gamebase|Json")
    UGamebaseJsonObject* AsObject();


    //////////////////////////////////////////////////////////////////////////
    // Data

private:
    /** Internal JSON data */
    TSharedPtr<FJsonValue> JsonVal;


    //////////////////////////////////////////////////////////////////////////
    // Helpers

protected:
    /** Simple error logger */
    void ErrorMessage(const FString& InType) const;

};