#pragma once

#include "CoreMinimal.h"
#include "GamebaseValueObject.h"

struct FGamebaseError;

using FGamebaseErrorPtr = TSharedPtr<FGamebaseError, ESPMode::ThreadSafe>;
using FGamebaseErrorRef = TSharedRef<FGamebaseError, ESPMode::ThreadSafe>;

struct GAMEBASE_API FGamebaseError final
    : FGamebaseValueObject
{
    FString Domain;
    int32 Code;
    FString Message;
    TOptional<FString> TransactionId;
    TOptional<TMap<FString, FString>> Extras;
    FGamebaseErrorPtr Error;

    FGamebaseError() = default;
    
    explicit FGamebaseError(const int32 Code, const FString& Message, const FString& Domain, const TOptional<FString>& TransactionId = {})
        : Domain(Domain)
        , Code(Code)
        , Message(Message)
        , TransactionId(TransactionId)
    {
    }
    
    explicit FGamebaseError(const int32 Code, const FString& Message, const FName& Domain, const TOptional<FString>& TransactionId = {})
        : Domain(Domain.ToString())
        , Code(Code)
        , Message(Message)
        , TransactionId(TransactionId)
    {
    }
    
    explicit FGamebaseError(const int32 Code, const FString& Message, const FString& Domain, const FGamebaseError& InnerError, const TOptional<FString>& TransactionId = {})
        : Domain(Domain)
        , Code(Code)
        , Message(Message)
        , TransactionId(TransactionId)
        , Error(MakeShared<FGamebaseError, ESPMode::ThreadSafe>(InnerError))
    {
    }
    
    explicit FGamebaseError(const int32 Code, const FString& Message, const FName& Domain, const FGamebaseError& InnerError, const TOptional<FString>& TransactionId = {})
        : Domain(Domain.ToString())
        , Code(Code)
        , Message(Message)
        , TransactionId(TransactionId)
        , Error(MakeShared<FGamebaseError, ESPMode::ThreadSafe>(InnerError))
    {
    }

    void AddExtra(const FString& Key, const FString& Value);
    TOptional<FString> FindExtra(const FString& Key) const;

    using FErrorVisitor = TFunction<void(const FGamebaseError&)>;
    void ForEachDepth(const FErrorVisitor& Visitor) const;
    int32 GetDepth() const;
    
    virtual void Serialize(FJsonSerializerBase& Serializer, bool bFlatObject) override;
};

DECLARE_DELEGATE_OneParam(FGamebaseErrorDelegate, const FGamebaseError*);
