#pragma once

#include "Misc/Variant.h"

class FGamebaseVariant;
class FGamebaseVariantMap;
class FGamebaseVariantArray;

enum class EGamebaseVariantType : int32
{
    Empty = 0,
    Bool,
    Int32,
    Int64,
    UInt32,
    UInt64,
    Double,
    Float,
    Name,
    String,
    Array,
    Object,
};

template<typename T> struct TGamebaseVariantTraits
{
    static CONSTEXPR EGamebaseVariantType GetType()
    {
        static_assert(!sizeof(T), "GamebaseVariant trait must be specialized for this type.");
        return EGamebaseVariantType::Empty;
    }
    
    static CONSTEXPR EVariantTypes GetVariantType()
    {
        static_assert(!sizeof(T), "Variant trait must be specialized for this type.");
        return EVariantTypes::Empty;
    }
};

template<> struct TGamebaseVariantTraits<bool>
{
    static CONSTEXPR EGamebaseVariantType GetType() { return EGamebaseVariantType::Bool; }
    static CONSTEXPR EVariantTypes GetVariantType() { return EVariantTypes::Bool; }
};

template<> struct TGamebaseVariantTraits<int32>
{
    static CONSTEXPR EGamebaseVariantType GetType() { return EGamebaseVariantType::Int32; }
    static CONSTEXPR EVariantTypes GetVariantType() { return EVariantTypes::Int32; }
};

template<> struct TGamebaseVariantTraits<int64>
{
    static CONSTEXPR EGamebaseVariantType GetType() { return EGamebaseVariantType::Int64; }
    static CONSTEXPR EVariantTypes GetVariantType() { return EVariantTypes::Int64; }
};

template<> struct TGamebaseVariantTraits<uint32>
{
    static CONSTEXPR EGamebaseVariantType GetType() { return EGamebaseVariantType::UInt32; }
    static CONSTEXPR EVariantTypes GetVariantType() { return EVariantTypes::UInt32; }
};

template<> struct TGamebaseVariantTraits<uint64>
{
    static CONSTEXPR EGamebaseVariantType GetType() { return EGamebaseVariantType::UInt64; }
    static CONSTEXPR EVariantTypes GetVariantType() { return EVariantTypes::UInt64; }
};

template<> struct TGamebaseVariantTraits<double>
{
    static CONSTEXPR EGamebaseVariantType GetType() { return EGamebaseVariantType::Double; }
    static CONSTEXPR EVariantTypes GetVariantType() { return EVariantTypes::Double; }
};

template<> struct TGamebaseVariantTraits<float>
{
    static CONSTEXPR EGamebaseVariantType GetType() { return EGamebaseVariantType::Float; }
    static CONSTEXPR EVariantTypes GetVariantType() { return EVariantTypes::Float; }
};

template<> struct TGamebaseVariantTraits<FName>
{
    static CONSTEXPR EGamebaseVariantType GetType() { return EGamebaseVariantType::Name; }
    static CONSTEXPR EVariantTypes GetVariantType() { return EVariantTypes::Name; }
};

template<> struct TGamebaseVariantTraits<FString>
{
    static CONSTEXPR EGamebaseVariantType GetType() { return EGamebaseVariantType::String; }
    static CONSTEXPR EVariantTypes GetVariantType() { return EVariantTypes::String; }
};

template<> struct TGamebaseVariantTraits<const wchar_t*>
{
    static CONSTEXPR EGamebaseVariantType GetType() { return EGamebaseVariantType::String; }
    static CONSTEXPR EVariantTypes GetVariantType() { return EVariantTypes::String; }
};

template<size_t N> struct TGamebaseVariantTraits<const wchar_t[N]>
{
    static CONSTEXPR EGamebaseVariantType GetType() { return EGamebaseVariantType::String; }
    static CONSTEXPR EVariantTypes GetVariantType() { return EVariantTypes::String; }
};

template<> struct TGamebaseVariantTraits<const char16_t*>
{
    static CONSTEXPR EGamebaseVariantType GetType() { return EGamebaseVariantType::String; }
    static CONSTEXPR EVariantTypes GetVariantType() { return EVariantTypes::String; }
};

template<size_t N> struct TGamebaseVariantTraits<const char16_t[N]>
{
    static CONSTEXPR EGamebaseVariantType GetType() { return EGamebaseVariantType::String; }
    static CONSTEXPR EVariantTypes GetVariantType() { return EVariantTypes::String; }
};

template<> struct TGamebaseVariantTraits<FGamebaseVariantArray>
{
    static CONSTEXPR EGamebaseVariantType GetType() { return EGamebaseVariantType::Array; }
    static CONSTEXPR EVariantTypes GetVariantType() { return EVariantTypes::ByteArray; }
};

template<> struct TGamebaseVariantTraits<FGamebaseVariantMap>
{
    static CONSTEXPR EGamebaseVariantType GetType() { return EGamebaseVariantType::Object; }
    static CONSTEXPR EVariantTypes GetVariantType() { return EVariantTypes::ByteArray; }
};

template<> struct TGamebaseVariantTraits<std::nullptr_t>
{
    static CONSTEXPR EGamebaseVariantType GetType() { return EGamebaseVariantType::Empty; }
    static CONSTEXPR EVariantTypes GetVariantType() { return EVariantTypes::Empty; }
};

template<> struct TGamebaseVariantTraits<FVariant>
{
    static CONSTEXPR EGamebaseVariantType GetType() { return EGamebaseVariantType::Empty; }
    static CONSTEXPR EVariantTypes GetVariantType() { return EVariantTypes::Custom; }
};

class FGamebaseVariant
{
public:
    FGamebaseVariant()
        : Type(EGamebaseVariantType::Empty)
    {
    }

    FGamebaseVariant(const FGamebaseVariant& Other)
        : Type(Other.Type)
        , Value(Other.Value)
    {
    }

    FGamebaseVariant& operator=(const FGamebaseVariant& Other)
    {
        if (this != &Other)
        {
            Type = Other.Type;
            Value = Other.Value;
        }
        return *this;
    }

    ~FGamebaseVariant() {}

    FGamebaseVariant(FGamebaseVariant&& Other)
        : Type(Other.Type)
        , Value(MoveTemp(Other.Value))
    {
        Other.Type = EGamebaseVariantType::Empty;
        Other.Value.Empty();
    }

    FGamebaseVariant& operator=(FGamebaseVariant&& Other)
    {
        if (this != &Other)
        {
            Type = Other.Type;
            Value = MoveTemp(Other.Value);
            Other.Type = EGamebaseVariantType::Empty;
            Other.Value.Empty();
        }
        return *this;
    }
    
    explicit FGamebaseVariant(const wchar_t* const& InValue)
        : Type(EGamebaseVariantType::String)
    {
        FString Str = FString(reinterpret_cast<const TCHAR*>(InValue));
        FMemoryWriter Writer(Value, true);
        Writer << Str;
    }
    
    explicit FGamebaseVariant(const char16_t* const& InValue)
        : Type(EGamebaseVariantType::String)
    {
        FString Str;
        if (InValue)
        {
            Str = FString(reinterpret_cast<const TCHAR*>(InValue));
        }
        FMemoryWriter Writer(Value, true);
        Writer << Str;
    }
    
    template<size_t N>
    explicit FGamebaseVariant(const wchar_t(&InValue)[N])
        : Type(EGamebaseVariantType::String)
    {
        FString Str = FString(reinterpret_cast<const TCHAR*>(InValue));
        FMemoryWriter Writer(Value, true);
        Writer << Str;
    }
    
    template<size_t N>
    explicit FGamebaseVariant(const char16_t(&InValue)[N])
        : Type(EGamebaseVariantType::String)
    {
        FString Str;
        if (InValue)
        {
            Str = FString(reinterpret_cast<const TCHAR*>(InValue));
        }
        FMemoryWriter Writer(Value, true);
        Writer << Str;
    }
    
    template<typename T, std::enable_if_t<!std::is_array<T>::value || (!std::is_same<std::remove_extent_t<T>, char16_t>::value && !std::is_same<std::remove_extent_t<T>, wchar_t>::value), int> = 0>
    explicit FGamebaseVariant(const T& InValue)
        : Type(TGamebaseVariantTraits<T>::GetType())
    {
        FMemoryWriter Writer(Value, true);
        Writer << const_cast<T&>(InValue);
    }

    template<typename T>
    FGamebaseVariant& operator=(const T& InValue)
    {
        Value.Empty();
        FMemoryWriter Writer(Value, true);
        Writer << const_cast<T&>(InValue);
        Type = TGamebaseVariantTraits<T>::GetType();
        return *this;
    }

    explicit FGamebaseVariant(std::nullptr_t)
        : Type(EGamebaseVariantType::Empty)
    {
    }

    explicit FGamebaseVariant(const FVariant& InValue)
    {
        switch (InValue.GetType())
        {
        case EVariantTypes::Bool:
            *this = InValue.GetValue<bool>();
            break;
        case EVariantTypes::Int32:
            *this = InValue.GetValue<int32>();
            break;
        case EVariantTypes::Int64:
            *this = InValue.GetValue<int64>();
            break;
        case EVariantTypes::UInt32:
            *this = InValue.GetValue<uint32>();
            break;
        case EVariantTypes::UInt64:
            *this = InValue.GetValue<uint64>();
            break;
        case EVariantTypes::Double:
            *this = InValue.GetValue<double>();
            break;
        case EVariantTypes::Float:
            *this = InValue.GetValue<float>();
            break;
        case EVariantTypes::Name:
            *this = InValue.GetValue<FName>();
            break;
        case EVariantTypes::String:
            *this = InValue.GetValue<FString>();
            break;
        case EVariantTypes::ByteArray:
            ensureMsgf(false, TEXT("FVariant ByteArray type is not supported."));
            Type = EGamebaseVariantType::Empty;
            break;
        case EVariantTypes::Empty:
            Type = EGamebaseVariantType::Empty;
            break;
        default:
            ensureMsgf(false, TEXT("Unsupported FVariant type: %d"), static_cast<int32>(InValue.GetType()));
            Type = EGamebaseVariantType::Empty;
            break;
        }
    }

    FORCEINLINE EGamebaseVariantType GetType() const
    {
        return Type;
    }

    FORCEINLINE bool IsEmpty() const
    {
        return Type == EGamebaseVariantType::Empty;
    }

    template<typename T>
    T GetValue() const
    {
        if (Type == TGamebaseVariantTraits<T>::GetType())
        {
            T Result;
            FMemoryReader Reader(Value, true);
            Reader << Result;
            return Result;
        }
        return T();
    }

    template<typename T>
    operator T() const
    {
        return GetValue<T>();
    }

    bool operator==(const FGamebaseVariant& Other) const
    {
        return Type == Other.Type && Value == Other.Value;
    }

    bool operator!=(const FGamebaseVariant& Other) const
    {
        return !(*this == Other);
    }

    friend FArchive& operator<<(FArchive& Ar, FGamebaseVariant& InVar)
    {
        Ar << InVar.Type;
        Ar << InVar.Value;
        return Ar;
    }

private:
    EGamebaseVariantType Type;
    TArray<uint8> Value;
};

class FGamebaseVariantMap
{
public:
    // Core type definitions for TMap compatibility
    using FKeyType = FName;
    using FValueType = FGamebaseVariant;
    using FVariantMapType = TMap<FKeyType, FValueType>;

    // Default constructors and operators
    FGamebaseVariantMap() = default;
    FGamebaseVariantMap(FGamebaseVariantMap&&) = default;
    FGamebaseVariantMap(const FGamebaseVariantMap&) = default;
    FGamebaseVariantMap& operator=(FGamebaseVariantMap&&) = default;
    FGamebaseVariantMap& operator=(const FGamebaseVariantMap&) = default;

    // Construction from TMap
    explicit FGamebaseVariantMap(const FVariantMapType& InMap) : Map(InMap) {}
    explicit FGamebaseVariantMap(FVariantMapType&& InMap) : Map(MoveTemp(InMap)) {}

    // Basic initializer_list constructors
    FGamebaseVariantMap(std::initializer_list<TPair<FKeyType, FValueType>> InitList)
    {
        Reserve(static_cast<int32>(InitList.size()));
        for (const auto& Pair : InitList)
        {
            Add(Pair.Key, Pair.Value);
        }
    }
    
    FGamebaseVariantMap(std::initializer_list<TPair<FKeyType, FString>> InitList)
    {
        Reserve(static_cast<int32>(InitList.size()));
        for (const auto& Pair : InitList)
        {
            Add(Pair.Key, FGamebaseVariant(Pair.Value));
        }
    }
    
    FGamebaseVariantMap(std::initializer_list<std::pair<FKeyType, FString>> InitList)
    {
        Reserve(static_cast<int32>(InitList.size()));
        for (const auto& Pair : InitList)
        {
            Add(Pair.first, FGamebaseVariant(Pair.second));
        }
    }
    
    FGamebaseVariantMap(std::initializer_list<std::pair<FKeyType, FName>> InitList)
    {
        Reserve(static_cast<int32>(InitList.size()));
        for (const auto& Pair : InitList)
        {
            Add(Pair.first, FValueType(Pair.second.ToString()));
        }
    }
    
    template<typename T, typename = std::enable_if_t<!std::is_same<T, FValueType>::value>>
    FGamebaseVariantMap(std::initializer_list<std::pair<FKeyType, T>> InitList)
    {
        Reserve(static_cast<int32>(InitList.size()));
        for (const auto& Pair : InitList)
        {
            Add(Pair.first, FValueType(Pair.second));
        }
    }
    
    template<typename T>
    FGamebaseVariantMap(std::initializer_list<std::pair<FKeyType, T>> InitList)
    {
        Reserve(static_cast<int32>(InitList.size()));
        for (const auto& Pair : InitList)
        {
            Add(Pair.first, FValueType(Pair.second));
        }
    }
    template<typename T>
    FGamebaseVariantMap(std::initializer_list<std::pair<const wchar_t*, T>> InitList)
    {
        Reserve(static_cast<int32>(InitList.size()));
        for (const auto& Pair : InitList)
        {
            Add(FKeyType(Pair.first), FValueType(Pair.second));
        }
    }
    template<typename T>
    FGamebaseVariantMap(std::initializer_list<std::pair<FString, T>> InitList)
    {
        Reserve(static_cast<int32>(InitList.size()));
        for (const auto& Pair : InitList)
        {
            Add(FKeyType(Pair.first), FValueType(Pair.second));
        }
    }
    
    template<typename KeyType, typename ValueType>
    FGamebaseVariantMap(std::initializer_list<std::pair<KeyType, ValueType>> InitList)
    {
        Reserve(static_cast<int32>(InitList.size()));
        for (const auto& Pair : InitList)
        {
            Add(FKeyType(Pair.first), FValueType(Pair.second));
        }
    }
    
    FGamebaseVariantMap(std::initializer_list<std::initializer_list<FKeyType>> InitList)
    {
        for (const auto& Pair : InitList)
        {
            auto It = Pair.begin();
            if (It != Pair.end())
            {
                const FKeyType Key = *It++;
                if (It != Pair.end())
                {
                    const FKeyType Value = *It;
                    Add(Key, FValueType(Value.ToString()));
                }
            }
        }
    }
    
    template<typename... Args>
    FGamebaseVariantMap(std::initializer_list<std::initializer_list<Args...>> InitList)
    {
        for (const auto& Pair : InitList)
        {
            auto It = Pair.begin();
            if (It != Pair.end())
            {
                const auto Key = *It++;
                if (It != Pair.end())
                {
                    const auto Value = *It;
                    Add(FKeyType(Key), FValueType(Value));
                }
            }
        }
    }
    
    // Core TMap-compatible methods
    FORCEINLINE void Empty(const int32 ExpectedNumElements = 0) { Map.Empty(ExpectedNumElements); }
    FORCEINLINE void Reset() { Map.Reset(); }
    FORCEINLINE void Shrink() { Map.Shrink(); }
    FORCEINLINE void Compact() { Map.Compact(); }
    FORCEINLINE void CompactStable() { Map.CompactStable(); }
    FORCEINLINE void Reserve(const int32 Number) { Map.Reserve(Number); }
    FORCEINLINE int32 Num() const { return Map.Num(); }

    // Add methods
    FORCEINLINE FValueType& Add(const FKeyType& InKey, const FValueType& InValue) { return Map.Add(InKey, InValue); }
    FORCEINLINE FValueType& Add(const FKeyType& InKey, FValueType&& InValue) { return Map.Add(InKey, MoveTempIfPossible(InValue)); }
    FORCEINLINE FValueType& Add(FKeyType&& InKey, const FValueType& InValue) { return Map.Add(MoveTempIfPossible(InKey), InValue); }
    FORCEINLINE FValueType& Add(FKeyType&& InKey, FValueType&& InValue) { return Map.Add(MoveTempIfPossible(InKey), MoveTempIfPossible(InValue)); }
    FORCEINLINE FValueType& Add(const FKeyType& InKey) { return Map.Add(InKey); }
    FORCEINLINE FValueType& Add(FKeyType&& InKey) { return Map.Add(MoveTempIfPossible(InKey)); }
    
    template<typename T>
    FValueType& Add(const FKeyType& InKey, T&& InValue)
    {
        return Map.Add(InKey, FValueType(Forward<T>(InValue)));
    }

    // Emplace methods
    template <typename InitKeyType, typename InitValueType>
    FValueType& Emplace(InitKeyType&& InKey, InitValueType&& InValue)
    {
        return Map.Emplace(Forward<InitKeyType>(InKey), Forward<InitValueType>(InValue));
    }

    template <typename InitKeyType>
    FValueType& Emplace(InitKeyType&& InKey)
    {
        return Map.Emplace(Forward<InitKeyType>(InKey));
    }

    // Other TMap methods
    FORCEINLINE int32 Remove(const FKeyType& InKey) { return Map.Remove(InKey); }
    FORCEINLINE bool RemoveAndCopyValue(const FKeyType& Key, FValueType& OutRemovedValue) { return Map.RemoveAndCopyValue(Key, OutRemovedValue); }
    FORCEINLINE FValueType FindAndRemoveChecked(const FKeyType& Key) { return Map.FindAndRemoveChecked(Key); }
    
    FORCEINLINE FValueType* Find(const FKeyType& Key) { return Map.Find(Key); }
    FORCEINLINE const FValueType* Find(const FKeyType& Key) const { return Map.Find(Key); }
    
    FORCEINLINE FValueType& FindOrAdd(const FKeyType& Key) { return Map.FindOrAdd(Key); }
    FORCEINLINE FValueType& FindOrAdd(FKeyType&& Key) { return Map.FindOrAdd(MoveTempIfPossible(Key)); }
    FORCEINLINE FValueType& FindOrAdd(const FKeyType& Key, const FValueType& Value) { return Map.FindOrAdd(Key, Value); }
    FORCEINLINE FValueType& FindOrAdd(const FKeyType& Key, FValueType&& Value) { return Map.FindOrAdd(Key, MoveTempIfPossible(Value)); }
    FORCEINLINE FValueType& FindOrAdd(FKeyType&& Key, const FValueType& Value) { return Map.FindOrAdd(MoveTempIfPossible(Key), Value); }
    FORCEINLINE FValueType& FindOrAdd(FKeyType&& Key, FValueType&& Value) { return Map.FindOrAdd(MoveTempIfPossible(Key), MoveTempIfPossible(Value)); }
    
    FORCEINLINE const FValueType& FindChecked(const FKeyType& Key) const { return Map.FindChecked(Key); }
    FORCEINLINE FValueType& FindChecked(const FKeyType& Key) { return Map.FindChecked(Key); }
    FORCEINLINE FValueType FindRef(const FKeyType& Key) const { return Map.FindRef(Key); }
    FORCEINLINE bool Contains(const FKeyType& Key) const { return Map.Contains(Key); }

    // Array generation methods
    template<typename Allocator> 
    void GenerateKeyArray(TArray<FKeyType, Allocator>& OutArray) const { Map.GenerateKeyArray(OutArray); }
    
    template<typename Allocator> 
    void GenerateValueArray(TArray<FValueType, Allocator>& OutArray) const { Map.GenerateValueArray(OutArray); }
    
    template<typename Allocator> 
    int32 GetKeys(TArray<FKeyType, Allocator>& OutKeys) const { return Map.GetKeys(OutKeys); }
    
    template<typename Allocator> 
    int32 GetKeys(TSet<FKeyType, Allocator>& OutKeys) const { return Map.GetKeys(OutKeys); }

    // Memory management
    FORCEINLINE uint32 GetAllocatedSize() const 
    { 
        return Map.GetAllocatedSize();
    }

    FORCEINLINE void CountBytes(FArchive& Ar) const 
    { 
        Map.CountBytes(Ar);
    }

    // Comparison operators
    bool OrderIndependentCompareEqual(const FGamebaseVariantMap& Other) const
    {
        return Map.OrderIndependentCompareEqual(Other.Map);
    }

    // Append methods
    template<typename OtherSetAllocator>
    void Append(TMap<FKeyType, FValueType, OtherSetAllocator>& OtherMap) { Map.Append(MoveTemp(OtherMap)); }
    
    template<typename OtherSetAllocator>
    void Append(const TMap<FKeyType, FValueType, OtherSetAllocator>& OtherMap) { Map.Append(OtherMap); }
    
    void Append(const FGamebaseVariantMap& Other)
    {
        Map.Append(Other.Map);
    }
    
    void Append(FGamebaseVariantMap&& Other)
    {
        Map.Append(MoveTemp(Other.Map));
    }

    // Operators
    FORCEINLINE FValueType& operator[](const FKeyType& Key) { return FindChecked(Key); }
    FORCEINLINE const FValueType& operator[](const FKeyType& Key) const { return FindChecked(Key); }
    
    // Iterator support
    FORCEINLINE auto CreateIterator() { return Map.CreateIterator(); }
    FORCEINLINE auto CreateConstIterator() const { return Map.CreateConstIterator(); }
    
    // Range-based for loop support using TMap iterators
    FORCEINLINE auto begin() { return Map.begin(); }
    FORCEINLINE auto begin() const { return Map.begin(); }
    FORCEINLINE auto end() { return Map.end(); }
    FORCEINLINE auto end() const { return Map.end(); }

    // Legacy comparison operators
    friend bool LegacyCompareEqual(const FGamebaseVariantMap& A, const FGamebaseVariantMap& B)
    {
        return LegacyCompareEqual(A.Map, B.Map);
    }
    
    friend bool LegacyCompareNotEqual(const FGamebaseVariantMap& A, const FGamebaseVariantMap& B)
    {
        return !LegacyCompareEqual(A, B);
    }

    // TMap conversion operators
    explicit operator TMap<FKeyType, FValueType>&() { return Map; }
    explicit operator const TMap<FKeyType, FValueType>&() const { return Map; }
    const TMap<FKeyType, FValueType>& GetConstTMap() const { return Map; }
    TMap<FKeyType, FValueType>& GetTMap() { return Map; }
    FVariantMapType& GetMap() { return Map; }
    const FVariantMapType& GetMap() const { return Map; }
    
    FORCEINLINE friend FArchive& operator<<(FArchive& Ar, FGamebaseVariantMap& InMap)
    {
        Ar << InMap.Map;
        return Ar;
    }
    
    template<typename T>
    void SetOptionalIfExists(const FKeyType& Key, TOptional<T>& OutValue) const
    {
        if (const FGamebaseVariant* FoundValue = Find(Key))
        {
            if (FoundValue->GetType() == TGamebaseVariantTraits<T>::GetType())
            {
                OutValue = FoundValue->GetValue<T>();
            }
            else
            {
                OutValue.Reset();
            }
        }
        else
        {
            OutValue.Reset();
        }
    }
    
    template<typename T>
    T GetValueOrDefault(const FKeyType& Key, const T& DefaultValue) const
    {
        if (const FValueType* FoundValue = Map.Find(Key))
        {
            if (FoundValue->GetType() == TGamebaseVariantTraits<T>::GetType())
            {
                return FoundValue->GetValue<T>();
            }
        }
        return DefaultValue;
    }
    
private:
    FVariantMapType Map;
};

class FGamebaseVariantArray
{
public:
    using FValueType = FGamebaseVariant;
    using FArrayType = TArray<FValueType>;

    FGamebaseVariantArray() = default;
    FGamebaseVariantArray(const FGamebaseVariantArray&) = default;
    FGamebaseVariantArray(FGamebaseVariantArray&&) = default;
    FGamebaseVariantArray& operator=(const FGamebaseVariantArray&) = default;
    FGamebaseVariantArray& operator=(FGamebaseVariantArray&&) = default;

    explicit FGamebaseVariantArray(const FArrayType& InArray) : Array(InArray) {}
    explicit FGamebaseVariantArray(FArrayType&& InArray) : Array(MoveTemp(InArray)) {}

    FGamebaseVariantArray(std::initializer_list<FValueType> InitList)
        : Array(InitList)
    {}

    FORCEINLINE void Add(const FValueType& Value) { Array.Add(Value); }
    FORCEINLINE void Add(FValueType&& Value) { Array.Add(MoveTemp(Value)); }
    FORCEINLINE int32 Num() const { return Array.Num(); }
    FORCEINLINE void Empty(int32 Slack = 0) { Array.Empty(Slack); }
    FORCEINLINE void Reset() { Array.Reset(); }
    FORCEINLINE FValueType& operator[](int32 Index) { return Array[Index]; }
    FORCEINLINE const FValueType& operator[](int32 Index) const { return Array[Index]; }
    FORCEINLINE auto begin() { return Array.begin(); }
    FORCEINLINE auto end() { return Array.end(); }
    FORCEINLINE auto begin() const { return Array.begin(); }
    FORCEINLINE auto end() const { return Array.end(); }

    explicit operator FArrayType&() { return Array; }
    explicit operator const FArrayType&() const { return Array; }
    FArrayType& GetArray() { return Array; }
    const FArrayType& GetArray() const { return Array; }

    friend FArchive& operator<<(FArchive& Ar, FGamebaseVariantArray& InArray)
    {
        Ar << InArray.Array;
        return Ar;
    }

private:
    FArrayType Array;
};
