#pragma once

#include "CoreMinimal.h"
#include "GamebaseBaseTypes.h"
#include "GamebaseError.h"
#include "GamebaseValueObject.h"
#include "GamebaseTypes.h"
#include "Misc/Variant.h"

class UGamebaseJsonObject;

struct GAMEBASE_API FGamebaseGraceBanInfo final : FGamebaseValueObject
{
    struct FPaymentStatus final
    {
        int64 Amount;
        int32 Count;
        FString Currency;
    };
    
    struct FReleaseRuleCondition final
    {
        int64 Amount;
        int32 Count;
        FString Currency;
        FString ConditionType;
    };
    
    int64 GracePeriodDate;
    FString Message;
    TOptional<FPaymentStatus> PaymentStatus;
    TOptional<FReleaseRuleCondition> ReleaseRuleCondition;
    
    virtual void Serialize(FJsonSerializerBase& Serializer, bool bFlatObject) override;
};


using FGamebaseTemporaryWithdrawalInfoPtr = TSharedPtr<struct FGamebaseTemporaryWithdrawalInfo, ESPMode::ThreadSafe>;

struct GAMEBASE_API FGamebaseTemporaryWithdrawalInfo final : FGamebaseValueObject
{
    int64 GracePeriodDate;
    
    virtual void Serialize(FJsonSerializerBase& Serializer, bool bFlatObject) override;
};

struct GAMEBASE_API FGamebaseAuthMappingInfo final : FGamebaseValueObject
{
    FString AuthKey;
    FString AuthSystem;
    FString IdpCode;
    int64 RegDate;
    FString UserId;
    
    virtual void Serialize(FJsonSerializerBase& Serializer, bool bFlatObject) override;
};


using FGamebaseAuthTokenPtr = TSharedPtr<struct FGamebaseAuthToken, ESPMode::ThreadSafe>;

struct GAMEBASE_API FGamebaseAuthToken final : FGamebaseValueObject
{
    struct GAMEBASE_API FToken final : FGamebaseValueObject
    {
        FString SourceIdpCode;
        FString AccessToken;
        TOptional<FString> AccessTokenSecret;
        TOptional<FString> SubCode;
        TOptional<int32> ExpiresIn;
        TOptional<FGamebaseVariantMap> ExtraParams;
        
        virtual void Serialize(FJsonSerializerBase& Serializer, bool bFlatObject) override;
        
        FGamebaseVariantMap ToCredentialInfo() const;
    };
    
    struct GAMEBASE_API FMember final : FGamebaseValueObject
    {
        FString AppId;
        TArray<FGamebaseAuthMappingInfo> AuthList;
        int64 LastLoginDate;
        int64 RegDate;
        FString UserId;
        FString Valid;
        TOptional<FGamebaseTemporaryWithdrawalInfo> TemporaryWithdrawal;
        TOptional<FGamebaseGraceBanInfo> GraceBan;
        
        virtual void Serialize(FJsonSerializerBase& Serializer, bool bFlatObject) override;
    };
    
    FToken Token;
    FMember Member;
    
    virtual void Serialize(FJsonSerializerBase& Serializer, bool bFlatObject) override;
};


using FGamebaseAuthProviderProfilePtr = TSharedPtr<struct FGamebaseAuthProviderProfile, ESPMode::ThreadSafe>;

struct GAMEBASE_API FGamebaseAuthProviderProfile final
{
    UGamebaseJsonObject* Information;
};


using FGamebaseBanInfoPtr = TSharedPtr<struct FGamebaseBanInfo, ESPMode::ThreadSafe>;

struct GAMEBASE_API FGamebaseBanInfo final : FGamebaseValueObject
{
    FString UserId;
    FString BanType;
    int64 BeginDate;
    int64 EndDate;
    FString Message;
    FString CsInfo;
    FString CsUrl;
    
    static FGamebaseBanInfoPtr From(const FGamebaseError* Error);
    
    virtual void Serialize(FJsonSerializerBase& Serializer, bool bFlatObject) override;
};


using FGamebaseTransferAccountInfoPtr = TSharedPtr<struct FGamebaseTransferAccountInfo, ESPMode::ThreadSafe>;

struct GAMEBASE_API FGamebaseTransferAccountInfo final : FGamebaseValueObject
{
    struct FAccount final
    {
        FString Id;
        FString Password;
    };
    
    struct FCondition final
    {
        FString TransferAccountType;
        FString ExpirationType;
        int64 ExpirationDate;
    };
    
    FString IssuedType;
    FAccount Account;
    FCondition Condition;
    
    virtual void Serialize(FJsonSerializerBase& Serializer, bool bFlatObject) override;
};


using FGamebaseTransferAccountFailInfoPtr = TSharedPtr<struct FGamebaseTransferAccountFailInfo, ESPMode::ThreadSafe>;

struct GAMEBASE_API FGamebaseTransferAccountFailInfo final : FGamebaseValueObject
{
    FString AppId;
    FString Id;
    FString Status;
    int32 FailCount;
    int64 BlockEndDate;
    int64 RegDate;
    
    static FGamebaseTransferAccountFailInfoPtr From(const FGamebaseError* Error);

    virtual void Serialize(FJsonSerializerBase& Serializer, bool bFlatObject) override;
};


using FGamebaseForcingMappingTicketPtr = TSharedPtr<struct FGamebaseForcingMappingTicket, ESPMode::ThreadSafe>;

struct GAMEBASE_API FGamebaseForcingMappingTicket final : FGamebaseValueObject
{
    FString UserId;
    
    FString MappedUserId;

    /**
     * The status of mapped user.
     * Use this value if you want to restrict addMapping according to user status.
     *
     * See https://docs.toast.com/ko/Game/Gamebase/ko/api-guide/#member-valid-code
     * In ForcingMappingTicket, mappedUserValid cannot be 'D' or 'M'.
     *
     * Y : Normal user. ('Y'es)
     * D : Withdrawn user. ('D'eleted)
     * B : 'B'anned user.
     * T : Withdrawal-suspended user. ('T'emporaryWithdrawn)
     * P : Ban-suspended user. ('P'ostpone)
     * M : 'M'issing account
     */
    FString MappedUserValid;
    
    FString IdpCode;
    
    FString ForcingMappingKey;
    
    int64 ExpirationDate;
    
    FString AccessToken;

    TOptional<FString> SubCode;

    TOptional<FGamebaseVariantMap> ExtraParams;
    
    static FGamebaseForcingMappingTicketPtr From(const FGamebaseError* Error);
    
    virtual void Serialize(FJsonSerializerBase& Serializer, bool bFlatObject) override;
};

DECLARE_DELEGATE_TwoParams(FGamebaseAuthTokenDelegate, const FGamebaseAuthToken*, const FGamebaseError*);
DECLARE_DELEGATE_TwoParams(FGamebaseTransferAccountDelegate, const FGamebaseTransferAccountInfo*, const FGamebaseError*);
DECLARE_DELEGATE_TwoParams(FGamebaseTemporaryWithdrawalDelegate, const FGamebaseTemporaryWithdrawalInfo*, const FGamebaseError*);
DECLARE_DELEGATE_TwoParams(FGamebaseLastLoggedInProviderDelegate, FString, const FGamebaseError*);