#pragma once

#include "CoreMinimal.h"
#include "GamebaseError.h"
#include "Types/GamebaseDataContainer.h"
#include "Types/GamebaseTermsTypes.h"

class GAMEBASE_API IGamebaseTerms
{
public:
    virtual ~IGamebaseTerms() = default;

    /**
     * Displays the terms and conditions window on the screen.
     * If the user agrees to the terms and conditions, it registers the consent or not on the server.
     * If you agree to the terms and conditions calling showTermsView again does not display the terms and conditions and immediately returns a success callback.
     *
     * @param CloseCallback    After agreeing to the terms and conditions, when the terms and conditions window is closed, the user is notified by a callback.
     *                         The FGamebaseDataContainer object that comes as a callback can be converted to PushConfiguration and used in RegisterPush API after logging in.
     * 
     * Example Usage:
     * @code
     *  void ShowTermsViewSample()
     *  {
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->GetTerms()->ShowTermsView(
     *          FGamebaseDataContainerDelegate::CreateLambda([=](const FGamebaseDataContainer* dataContainer, const FGamebaseError* Error) {
     *              if (Gamebase::IsSuccess(Error))
     *              {
     *                  UE_LOG(GamebaseTestResults, Display, TEXT("RegisterPush succeeded."));
     *
     *                  // Save the PushConfiguration and use it for RegisterPush after Login.
     *                  const auto pushConfiguration = FGamebasePushConfiguration::From(dataContainer);
     *              }
     *              else
     *              {
     *                  UE_LOG(GamebaseTestResults, Display, TEXT("RegisterPush failed. (Error: %d)"), Error->code);
     *              }
     *          })
     *      );
     *  }
     * @endcode
     */
    virtual void ShowTermsView(const FGamebaseDataContainerDelegate& CloseCallback) = 0;
 
    /**
     * Displays the terms and conditions window on the screen.
     * If the user agrees to the terms and conditions, it registers the consent or not on the server.
     * If you agree to the terms and conditions calling showTermsView again does not display the terms and conditions and immediately returns a success callback.
     *
     * @param Configuration The initial settings of terms view.
     * @param CloseCallback    After agreeing to the terms and conditions, when the terms and conditions window is closed, the user is notified by a callback. The FGamebaseDataContainer object that comes as a callback can be converted to PushConfiguration and used in RegisterPush API after logging in.
     * 
     * Example Usage:
     * @code
     *  void ShowTermsViewSample(bool forceShow, bool bEnableFixedFontSize)
     *  {
     *      FGamebaseTermsConfiguration Configuration;
     *      Configuration.bForceShow = bForceShow;
     *      Configuration.bEnableFixedFontSize = bEnableFixedFontSize;
     *      
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->GetTerms()->ShowTermsView(Configuration,
     *          FGamebaseDataContainerDelegate::CreateLambda([=](const FGamebaseDataContainer* dataContainer, const FGamebaseError* Error) {
     *              if (Gamebase::IsSuccess(Error))
     *              {
     *                  UE_LOG(GamebaseTestResults, Display, TEXT("RegisterPush succeeded."));
     *
     *                  // Save the PushConfiguration and use it for RegisterPush after Login.
     *                  const auto pushConfiguration = FGamebasePushConfiguration::From(dataContainer);
     *              }
     *              else
     *              {
     *                  UE_LOG(GamebaseTestResults, Display, TEXT("RegisterPush failed. (Error: %d)"), Error->code);
     *              }
     *          })
     *      );
     *  }
     * @endcode
     */
    virtual void ShowTermsView(const FGamebaseTermsConfiguration& Configuration, const FGamebaseDataContainerDelegate& CloseCallback) = 0;

    
    /**
     * If you have created your own UI with the terms and conditions information downloaded through the queryTerms API,
     * Please send the game user's agreement to the terms and conditions to the Gamebase server through the updateTerms API.
     * You can also use it for the purpose of changing the details of your agreement to the terms, such as canceling the agreement to the optional terms and conditions.
     *
     * @param Configuration This is the information on the user's option to register on the server.
     * @param Callback      The optional terms and conditions are registered on the server and notified to the user via a callback.
     * 
     * Example Usage:
     * @code
     *  void UpdateTermsSample(int32 TermsSeq, const FString& TermsVersion, int32 TermsContentSeq, bool bAgreed)
     *  {
     *      FGamebaseTermsContent Content;
     *      Content.TermsContentSeq = TermsContentSeq;
     *      Content.bAgreed = bAgreed;
     *      
     *      FGamebaseUpdateTermsConfiguration Configuration;
     *      Configuration.TermsSeq = TermsSeq;
     *      Configuration.TermsVersion = TermsVersion;
     *      Configuration.Contents.Add(Content);
     *      
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->GetTerms()->UpdateTerms(Configuration,
     *          FGamebaseDataContainerDelegate::CreateLambda([=](const FGamebaseError* Error) {
     *              if (Gamebase::IsSuccess(Error))
     *              {
     *                  UE_LOG(GamebaseTestResults, Display, TEXT("UpdateTerms succeeded."));
     *              }
     *              else
     *              {
     *                  UE_LOG(GamebaseTestResults, Display, TEXT("UpdateTerms failed. (Error: %d)"), Error->code);
     *              }
     *          })
     *      );
     *  }
     * @endcode
     */
    virtual void UpdateTerms(const FGamebaseUpdateTermsConfiguration& Configuration, const FGamebaseErrorDelegate& Callback) = 0;
    
    /**
     * If you develop the terms and conditions for the game UI yourself, you can call queryTerms API to receive and use the terms and conditions set in the Gamebase console.
     * If you call the API after logging in, you can also check whether the game user has agreed to the terms and conditions.
     *
     * @param Callback    The result of the API call is notified to the user as a callback. You can get the terms and conditions set in the console with GamebaseQueryTermsResult that comes as a callback.
     * 
     * Example Usage:
     * @code
     *  void QueryTermsSample()
     *  {
     *      UGamebaseSubsystem* GamebaseSubsystem = UGameInstance::GetSubsystem<UGamebaseSubsystem>(GetGameInstance());
     *      GamebaseSubsystem->GetTerms()->QueryTerms(
     *          FGamebaseQueryTermsResultDelegate::CreateLambda([=](const FGamebaseQueryTermsResult* data, const FGamebaseError* Error) {
     *              if (Gamebase::IsSuccess(Error))
     *              {
     *                  UE_LOG(GamebaseTestResults, Display, TEXT("QueryTerms succeeded."));
     *              }
     *              else
     *              {
     *                  UE_LOG(GamebaseTestResults, Display, TEXT("QueryTerms failed. (Error: %d)"), Error->code);
     *              }
     *          })
     *      );
     *  }
     * @endcode
     */
    virtual void QueryTerms(const FGamebaseQueryTermsResultDelegate& Callback) = 0;

    /**
     * This is the method to check if terms view is being shown.
     *
     * @return Whether the terms view is currently show.
     */
    virtual bool IsShowingTermsView() = 0;
};