#include "GamebasePushTypes.h"

#include "GamebaseJsonTypes.h"
#include "Utilities/GamebaseValueObjectSerializerMacros.h"

namespace AgreePushTypes
{
    const FString None{ TEXT("NONE") };
    const FString Day{ TEXT("DAY") };
    const FString Night{ TEXT("NIGHT") };
    const FString All{ TEXT("ALL") };
}

FGamebasePushConfigurationPtr FGamebasePushConfiguration::From(const FGamebaseDataContainer* DataContainer)
{
    if (DataContainer == nullptr || DataContainer->Data.IsEmpty())
    {
        return nullptr;
    }
    
    UGamebaseJsonObject* JsonObject = NewObject<UGamebaseJsonObject>();
    if (JsonObject->DecodeJson(DataContainer->Data) == false)
    {
        return nullptr;
    }

    const FString AgreePush = JsonObject->GetStringField("agreePush");
    
    FGamebasePushConfigurationPtr PushConfiguration = MakeShared<FGamebasePushConfiguration, ESPMode::ThreadSafe>();
    if (AgreePush.Equals(AgreePushTypes::Day) == true)
    {
        PushConfiguration->bPushEnabled = true;
        PushConfiguration->bADAgreement = true;
        PushConfiguration->bADAgreementNight = false;
    }
    if (AgreePush.Equals(AgreePushTypes::Night) == true)
    {
        PushConfiguration->bPushEnabled = true;
        PushConfiguration->bADAgreement = false;
        PushConfiguration->bADAgreementNight = true;
    }
    if (AgreePush.Equals(AgreePushTypes::All) == true)
    {
        PushConfiguration->bPushEnabled = true;
        PushConfiguration->bADAgreement = true;
        PushConfiguration->bADAgreementNight = true;
    }
    else
    {
        // 'agreePush' values handled in this case: NONE and other (including empty string)
        PushConfiguration->bPushEnabled = true;
        PushConfiguration->bADAgreement = false;
        PushConfiguration->bADAgreementNight = false;
    }

    return PushConfiguration;
}

void FGamebasePushConfiguration::Serialize(FJsonSerializerBase& Serializer, bool bFlatObject)
{
    BEGIN_GAMEBASE_VO_SERIALIZER
    GAMEBASE_VO_SERIALIZE("pushEnabled", bPushEnabled);
    GAMEBASE_VO_SERIALIZE("adAgreement", bADAgreement);
    GAMEBASE_VO_SERIALIZE("adAgreementNight", bADAgreementNight);
    GAMEBASE_VO_SERIALIZE("displayLanguageCode", DisplayLanguageCode);
    GAMEBASE_VO_SERIALIZE_OPTIONAL("requestNotificationPermission", bRequestNotificationPermission);
    GAMEBASE_VO_SERIALIZE_OPTIONAL("alwaysAllowTokenRegistration", bAlwaysAllowTokenRegistration);
    END_GAMEBASE_VO_SERIALIZER
}

void FGamebaseNotificationOptions::Serialize(FJsonSerializerBase& Serializer, bool bFlatObject)
{
    BEGIN_GAMEBASE_VO_SERIALIZER
    GAMEBASE_VO_SERIALIZE("foregroundEnabled", bForegroundEnabled);
    GAMEBASE_VO_SERIALIZE("badgeEnabled", bBadgeEnabled);
    GAMEBASE_VO_SERIALIZE("soundEnabled", bSoundEnabled);
    GAMEBASE_VO_SERIALIZE("priority", Priority);
    GAMEBASE_VO_SERIALIZE("smallIconName", SmallIconName);
    GAMEBASE_VO_SERIALIZE("soundFileName", SoundFileName);
    END_GAMEBASE_VO_SERIALIZER
}

void FGamebasePushTokenInfo::Serialize(FJsonSerializerBase& Serializer, bool bFlatObject)
{
    auto AgreementStatusSerialize = [](FAgreement& ResultType, FJsonSerializerBase& Serializer, bool bFlatObject)
    {
        BEGIN_GAMEBASE_VO_SERIALIZER
        GAMEBASE_VO_SERIALIZE_IN("pushEnabled", bPushEnabled);
        GAMEBASE_VO_SERIALIZE_IN("adAgreement", bAdAgreement);
        GAMEBASE_VO_SERIALIZE_IN("adAgreementNight", bAdAgreementNight);
        END_GAMEBASE_VO_SERIALIZER
    };
    
    BEGIN_GAMEBASE_VO_SERIALIZER
    GAMEBASE_VO_SERIALIZE("sandbox", bSandbox);
    GAMEBASE_VO_SERIALIZE("pushType", PushType);
    GAMEBASE_VO_SERIALIZE("token", Token);
    GAMEBASE_VO_SERIALIZE("userId", UserId);
    GAMEBASE_VO_SERIALIZE("deviceCountryCode", DeviceCountryCode);
    GAMEBASE_VO_SERIALIZE("timezone", Timezone);
    GAMEBASE_VO_SERIALIZE("registeredDateTime", RegisteredDateTime);
    GAMEBASE_VO_SERIALIZE("languageCode", LanguageCode);
    GAMEBASE_VO_SERIALIZE_OBJECT_SERIALIZABLE_INNER("agreement", Agreement, AgreementStatusSerialize);
    END_GAMEBASE_VO_SERIALIZER
}
