#include "GamebaseJsonTypes.h"

#include "Policies/CondensedJsonPrintPolicy.h"
#include "Serialization/JsonWriter.h"
#include "Serialization/JsonSerializer.h"

typedef TJsonWriterFactory< TCHAR, TCondensedJsonPrintPolicy<TCHAR> > FCondensedJsonStringWriterFactory;
typedef TJsonWriter< TCHAR, TCondensedJsonPrintPolicy<TCHAR> > FCondensedJsonStringWriter;

UGamebaseJsonObject::UGamebaseJsonObject(const FObjectInitializer& ObjectInitializer)
    : Super(ObjectInitializer)
{
    Reset();
}

UGamebaseJsonObject* UGamebaseJsonObject::ConstructJsonObject(UObject* WorldContextObject)
{
    return NewObject<UGamebaseJsonObject>();
}

void UGamebaseJsonObject::Reset()
{
    if (JsonObj.IsValid())
    {
        JsonObj.Reset();
    }

    JsonObj = MakeShareable(new FJsonObject());
}

const TSharedPtr<FJsonObject>& UGamebaseJsonObject::GetRootObject() const
{
    return JsonObj;
}

void UGamebaseJsonObject::SetRootObject(const TSharedPtr<FJsonObject>& JsonObject)
{
    JsonObj = JsonObject;
}


//////////////////////////////////////////////////////////////////////////
// Serialization

FString UGamebaseJsonObject::EncodeJson() const
{
    if (!JsonObj.IsValid())
    {
        return TEXT("");
    }

    FString OutputString;
    TSharedRef< FCondensedJsonStringWriter > Writer = FCondensedJsonStringWriterFactory::Create(&OutputString);
    FJsonSerializer::Serialize(JsonObj.ToSharedRef(), Writer);

    return OutputString;
}

FString UGamebaseJsonObject::EncodeJsonToSingleString() const
{
    FString OutputString = EncodeJson();

    // Remove line terminators
    OutputString = OutputString.Replace(LINE_TERMINATOR, TEXT(""));

    // Remove tabs
    OutputString = OutputString.Replace(TEXT("\t"), TEXT(""));

    return OutputString;
}

bool UGamebaseJsonObject::DecodeJson(const FString& JsonString)
{
    TSharedRef< TJsonReader<> > Reader = TJsonReaderFactory<>::Create(*JsonString);
    if (FJsonSerializer::Deserialize(Reader, JsonObj) && JsonObj.IsValid())
    {
        return true;
    }

    // If we've failed to deserialize the string, we should clear our internal data
    Reset();

    //GAMEBASE_LOG_ERROR("Json decoding failed for: %s", *JsonString);

    return false;
}


//////////////////////////////////////////////////////////////////////////
// FJsonObject API

TArray<FString> UGamebaseJsonObject::GetFieldNames()
{
    TArray<FString> Result;

    if (!JsonObj.IsValid())
    {
        return Result;
    }

    JsonObj->Values.GetKeys(Result);

    return Result;
}

bool UGamebaseJsonObject::HasField(const FString& FieldName) const
{
    if (!JsonObj.IsValid() || FieldName.IsEmpty())
    {
        return false;
    }

    return JsonObj->HasField(FieldName);
}

void UGamebaseJsonObject::RemoveField(const FString& FieldName)
{
    if (!JsonObj.IsValid() || FieldName.IsEmpty())
    {
        return;
    }

    JsonObj->RemoveField(FieldName);
}

UGamebaseJsonValue* UGamebaseJsonObject::GetField(const FString& FieldName) const
{
    if (!JsonObj.IsValid() || FieldName.IsEmpty())
    {
        return nullptr;
    }

    TSharedPtr<FJsonValue> NewVal = JsonObj->TryGetField(FieldName);
    if (NewVal.IsValid())
    {
        UGamebaseJsonValue* NewValue = NewObject<UGamebaseJsonValue>();
        NewValue->SetRootValue(NewVal);

        return NewValue;
    }

    return nullptr;
}

void UGamebaseJsonObject::SetField(const FString& FieldName, UGamebaseJsonValue* JsonValue)
{
    if (!JsonObj.IsValid() || FieldName.IsEmpty())
    {
        return;
    }

    JsonObj->SetField(FieldName, JsonValue->GetRootValue());
}


void UGamebaseJsonObject::SetFieldNull(const FString& FieldName)
{
    if (!JsonObj.IsValid())
    {
        return;
    }

    TSharedPtr<FJsonValue> myNull = MakeShareable(new FJsonValueNull());

    JsonObj->SetField(FieldName, myNull);
}

//////////////////////////////////////////////////////////////////////////
// FJsonObject API Helpers (easy to use with simple Json objects)

float UGamebaseJsonObject::GetNumberField(const FString& FieldName) const
{
    if (!JsonObj.IsValid() || !JsonObj->HasTypedField<EJson::Number>(FieldName))
    {
        //GAMEBASE_LOG_WARNING("No field with name %s of type Number", *FieldName);
        return 0.0f;
    }

    return JsonObj->GetNumberField(FieldName);
}

void UGamebaseJsonObject::SetNumberField(const FString& FieldName, float Number)
{
    if (!JsonObj.IsValid() || FieldName.IsEmpty())
    {
        return;
    }

    JsonObj->SetNumberField(FieldName, Number);
}

FString UGamebaseJsonObject::GetStringField(const FString& FieldName) const
{
    if (!JsonObj.IsValid() || !JsonObj->HasTypedField<EJson::String>(FieldName))
    {
        //GAMEBASE_LOG_WARNING("No field with name %s of type String", *FieldName);
        return TEXT("");
    }

    return JsonObj->GetStringField(FieldName);
}

void UGamebaseJsonObject::SetStringField(const FString& FieldName, const FString& StringValue)
{
    if (!JsonObj.IsValid() || FieldName.IsEmpty())
    {
        return;
    }

    JsonObj->SetStringField(FieldName, StringValue);
}

bool UGamebaseJsonObject::GetBoolField(const FString& FieldName) const
{
    if (!JsonObj.IsValid() || !JsonObj->HasTypedField<EJson::Boolean>(FieldName))
    {
        //GAMEBASE_LOG_WARNING("No field with name %s of type Boolean", *FieldName);
        return false;
    }

    return JsonObj->GetBoolField(FieldName);
}

void UGamebaseJsonObject::SetBoolField(const FString& FieldName, bool InValue)
{
    if (!JsonObj.IsValid() || FieldName.IsEmpty())
    {
        return;
    }

    JsonObj->SetBoolField(FieldName, InValue);
}

TArray<UGamebaseJsonValue*> UGamebaseJsonObject::GetArrayField(const FString& FieldName)
{
    if (!JsonObj->HasTypedField<EJson::Array>(FieldName))
    {
        //GAMEBASE_LOG_WARNING("No field with name %s of type Array", *FieldName);
    }

    TArray<UGamebaseJsonValue*> OutArray;
    if (!JsonObj.IsValid() || FieldName.IsEmpty())
    {
        return OutArray;
    }

    TArray< TSharedPtr<FJsonValue> > ValArray = JsonObj->GetArrayField(FieldName);
    for (auto Value : ValArray)
    {
        UGamebaseJsonValue* NewValue = NewObject<UGamebaseJsonValue>();
        NewValue->SetRootValue(Value);

        OutArray.Add(NewValue);
    }

    return OutArray;
}

void UGamebaseJsonObject::SetArrayField(const FString& FieldName, const TArray<UGamebaseJsonValue*>& InArray)
{
    if (!JsonObj.IsValid() || FieldName.IsEmpty())
    {
        return;
    }

    TArray< TSharedPtr<FJsonValue> > ValArray;

    // Process input array and COPY original values
    for (auto InVal : InArray)
    {
        TSharedPtr<FJsonValue> JsonVal = InVal->GetRootValue();

        switch (InVal->GetType())
        {
        case EGamebaseJson::None:
            break;

        case EGamebaseJson::Null:
            ValArray.Add(MakeShareable(new FJsonValueNull()));
            break;

        case EGamebaseJson::String:
            ValArray.Add(MakeShareable(new FJsonValueString(JsonVal->AsString())));
            break;

        case EGamebaseJson::Number:
            ValArray.Add(MakeShareable(new FJsonValueNumber(JsonVal->AsNumber())));
            break;

        case EGamebaseJson::Boolean:
            ValArray.Add(MakeShareable(new FJsonValueBoolean(JsonVal->AsBool())));
            break;

        case EGamebaseJson::Array:
            ValArray.Add(MakeShareable(new FJsonValueArray(JsonVal->AsArray())));
            break;

        case EGamebaseJson::Object:
            ValArray.Add(MakeShareable(new FJsonValueObject(JsonVal->AsObject())));
            break;

        default:
            break;
        }
    }

    JsonObj->SetArrayField(FieldName, ValArray);
}

void UGamebaseJsonObject::MergeJsonObject(UGamebaseJsonObject* InJsonObject, bool Overwrite)
{
    TArray<FString> Keys = InJsonObject->GetFieldNames();

    for (auto Key : Keys)
    {
        if (Overwrite == false && HasField(Key))
        {
            continue;
        }

        SetField(Key, InJsonObject->GetField(Key));
    }
}

UGamebaseJsonObject* UGamebaseJsonObject::GetObjectField(const FString& FieldName) const
{
    if (!JsonObj.IsValid() || !JsonObj->HasTypedField<EJson::Object>(FieldName))
    {
        //GAMEBASE_LOG_WARNING("No field with name %s of type Object", *FieldName);
        return nullptr;
    }

    TSharedPtr<FJsonObject> JsonObjField = JsonObj->GetObjectField(FieldName);

    UGamebaseJsonObject* OutRestJsonObj = NewObject<UGamebaseJsonObject>();
    OutRestJsonObj->SetRootObject(JsonObjField);

    return OutRestJsonObj;
}

void UGamebaseJsonObject::SetObjectField(const FString& FieldName, const UGamebaseJsonObject* JsonObject)
{
    if (!JsonObj.IsValid() || FieldName.IsEmpty())
    {
        return;
    }

    if (JsonObject != nullptr)
    {
        JsonObj->SetObjectField(FieldName, JsonObject->GetRootObject());
    }
    else
    {
        TSharedPtr<FJsonValue> myNull = MakeShareable(new FJsonValueNull());
        JsonObj->SetField(FieldName, myNull);
    }
}

//////////////////////////////////////////////////////////////////////////
// Array fields helpers (uniform arrays)

TArray<float> UGamebaseJsonObject::GetNumberArrayField(const FString& FieldName)
{
    if (!JsonObj->HasTypedField<EJson::Array>(FieldName))
    {
        //GAMEBASE_LOG_WARNING("No field with name %s of type Array", *FieldName);
    }

    TArray<float> NumberArray;
    if (!JsonObj.IsValid() || FieldName.IsEmpty())
    {
        return NumberArray;
    }

    TArray<TSharedPtr<FJsonValue> > JsonArrayValues = JsonObj->GetArrayField(FieldName);
    for (TArray<TSharedPtr<FJsonValue> >::TConstIterator It(JsonArrayValues); It; ++It)
    {
        auto Value = (*It).Get();
        if (Value->Type != EJson::Number)
        {
            //GAMEBASE_LOG_ERROR("Not Number element in array with field name %s", *FieldName);
        }

        NumberArray.Add((*It)->AsNumber());
    }

    return NumberArray;
}

void UGamebaseJsonObject::SetNumberArrayField(const FString& FieldName, const TArray<float>& NumberArray)
{
    if (!JsonObj.IsValid() || FieldName.IsEmpty())
    {
        return;
    }

    TArray< TSharedPtr<FJsonValue> > EntriesArray;

    for (auto Number : NumberArray)
    {
        EntriesArray.Add(MakeShareable(new FJsonValueNumber(Number)));
    }

    JsonObj->SetArrayField(FieldName, EntriesArray);
}

TArray<FString> UGamebaseJsonObject::GetStringArrayField(const FString& FieldName)
{
    if (!JsonObj->HasTypedField<EJson::Array>(FieldName))
    {
        //GAMEBASE_LOG_WARNING("No field with name %s of type Array", *FieldName);
    }

    TArray<FString> StringArray;
    if (!JsonObj.IsValid() || FieldName.IsEmpty())
    {
        return StringArray;
    }

    TArray<TSharedPtr<FJsonValue> > JsonArrayValues = JsonObj->GetArrayField(FieldName);
    for (TArray<TSharedPtr<FJsonValue> >::TConstIterator It(JsonArrayValues); It; ++It)
    {
        auto Value = (*It).Get();
        if (Value->Type != EJson::String)
        {
            //GAMEBASE_LOG_ERROR("Not String element in array with field name %s", *FieldName);
        }

        StringArray.Add((*It)->AsString());
    }

    return StringArray;
}

void UGamebaseJsonObject::SetStringArrayField(const FString& FieldName, const TArray<FString>& StringArray)
{
    if (!JsonObj.IsValid() || FieldName.IsEmpty())
    {
        return;
    }

    TArray< TSharedPtr<FJsonValue> > EntriesArray;
    for (auto String : StringArray)
    {
        EntriesArray.Add(MakeShareable(new FJsonValueString(String)));
    }

    JsonObj->SetArrayField(FieldName, EntriesArray);
}

TArray<bool> UGamebaseJsonObject::GetBoolArrayField(const FString& FieldName)
{
    if (!JsonObj->HasTypedField<EJson::Array>(FieldName))
    {
        //GAMEBASE_LOG_WARNING("No field with name %s of type Array", *FieldName);
    }

    TArray<bool> BoolArray;
    if (!JsonObj.IsValid() || FieldName.IsEmpty())
    {
        return BoolArray;
    }

    TArray<TSharedPtr<FJsonValue> > JsonArrayValues = JsonObj->GetArrayField(FieldName);
    for (TArray<TSharedPtr<FJsonValue> >::TConstIterator It(JsonArrayValues); It; ++It)
    {
        auto Value = (*It).Get();
        if (Value->Type != EJson::Boolean)
        {
            //GAMEBASE_LOG_ERROR("Not Boolean element in array with field name %s", *FieldName);
        }

        BoolArray.Add((*It)->AsBool());
    }

    return BoolArray;
}

void UGamebaseJsonObject::SetBoolArrayField(const FString& FieldName, const TArray<bool>& BoolArray)
{
    if (!JsonObj.IsValid() || FieldName.IsEmpty())
    {
        return;
    }

    TArray< TSharedPtr<FJsonValue> > EntriesArray;
    for (auto Boolean : BoolArray)
    {
        EntriesArray.Add(MakeShareable(new FJsonValueBoolean(Boolean)));
    }

    JsonObj->SetArrayField(FieldName, EntriesArray);
}

TArray<UGamebaseJsonObject*> UGamebaseJsonObject::GetObjectArrayField(const FString& FieldName)
{
    if (!JsonObj->HasTypedField<EJson::Array>(FieldName))
    {
        //GAMEBASE_LOG_WARNING("No field with name %s of type Array", *FieldName);
    }

    TArray<UGamebaseJsonObject*> OutArray;
    if (!JsonObj.IsValid() || FieldName.IsEmpty())
    {
        return OutArray;
    }

    TArray< TSharedPtr<FJsonValue> > ValArray = JsonObj->GetArrayField(FieldName);
    for (auto Value : ValArray)
    {
        if (Value->Type != EJson::Object)
        {
            //GAMEBASE_LOG_ERROR("Not Object element in array with field name %s", *FieldName);
        }

        TSharedPtr<FJsonObject> NewObj = Value->AsObject();

        UGamebaseJsonObject* NewJson = NewObject<UGamebaseJsonObject>();
        NewJson->SetRootObject(NewObj);

        OutArray.Add(NewJson);
    }

    return OutArray;
}

void UGamebaseJsonObject::SetObjectArrayField(const FString& FieldName, const TArray<UGamebaseJsonObject*>& ObjectArray)
{
    if (!JsonObj.IsValid() || FieldName.IsEmpty())
    {
        return;
    }

    TArray< TSharedPtr<FJsonValue> > EntriesArray;
    for (auto Value : ObjectArray)
    {
        EntriesArray.Add(MakeShareable(new FJsonValueObject(Value->GetRootObject())));
    }

    JsonObj->SetArrayField(FieldName, EntriesArray);
}


#if PLATFORM_WINDOWS
#pragma endregion FJsonValueBinary
#pragma region UGamebaseJsonValue
#endif

UGamebaseJsonValue::UGamebaseJsonValue(const class FObjectInitializer& PCIP)
    : Super(PCIP)
{

}

UGamebaseJsonValue* UGamebaseJsonValue::ConstructJsonValueNumber(UObject* WorldContextObject, float Number)
{
    TSharedPtr<FJsonValue> NewVal = MakeShareable(new FJsonValueNumber(Number));

    UGamebaseJsonValue* NewValue = NewObject<UGamebaseJsonValue>();
    NewValue->SetRootValue(NewVal);

    return NewValue;
}

UGamebaseJsonValue* UGamebaseJsonValue::ConstructJsonValueString(UObject* WorldContextObject, const FString& StringValue)
{
    TSharedPtr<FJsonValue> NewVal = MakeShareable(new FJsonValueString(StringValue));

    UGamebaseJsonValue* NewValue = NewObject<UGamebaseJsonValue>();
    NewValue->SetRootValue(NewVal);

    return NewValue;
}

UGamebaseJsonValue* UGamebaseJsonValue::ConstructJsonValueBool(UObject* WorldContextObject, bool InValue)
{
    TSharedPtr<FJsonValue> NewVal = MakeShareable(new FJsonValueBoolean(InValue));

    UGamebaseJsonValue* NewValue = NewObject<UGamebaseJsonValue>();
    NewValue->SetRootValue(NewVal);

    return NewValue;
}

UGamebaseJsonValue* UGamebaseJsonValue::ConstructJsonValueArray(UObject* WorldContextObject, const TArray<UGamebaseJsonValue*>& InArray)
{
    // Prepare data array to create new value
    TArray< TSharedPtr<FJsonValue> > ValueArray;
    for (auto InVal : InArray)
    {
        ValueArray.Add(InVal->GetRootValue());
    }

    TSharedPtr<FJsonValue> NewVal = MakeShareable(new FJsonValueArray(ValueArray));

    UGamebaseJsonValue* NewValue = NewObject<UGamebaseJsonValue>();
    NewValue->SetRootValue(NewVal);

    return NewValue;
}

UGamebaseJsonValue* UGamebaseJsonValue::ConstructJsonValueObject(UGamebaseJsonObject *JsonObject, UObject* WorldContextObject)
{
    TSharedPtr<FJsonValue> NewVal = MakeShareable(new FJsonValueObject(JsonObject->GetRootObject()));

    UGamebaseJsonValue* NewValue = NewObject<UGamebaseJsonValue>();
    NewValue->SetRootValue(NewVal);

    return NewValue;
}

UGamebaseJsonValue* UGamebaseJsonValue::ConstructJsonValue(UObject* WorldContextObject, const TSharedPtr<FJsonValue>& InValue)
{
    TSharedPtr<FJsonValue> NewVal = InValue;

    UGamebaseJsonValue* NewValue = NewObject<UGamebaseJsonValue>();
    NewValue->SetRootValue(NewVal);

    return NewValue;
}

TSharedPtr<FJsonValue>& UGamebaseJsonValue::GetRootValue()
{
    return JsonVal;
}

void UGamebaseJsonValue::SetRootValue(TSharedPtr<FJsonValue>& JsonValue)
{
    JsonVal = JsonValue;
}


//////////////////////////////////////////////////////////////////////////
// FJsonValue API

EGamebaseJson::Type UGamebaseJsonValue::GetType() const
{
    if (!JsonVal.IsValid())
    {
        return EGamebaseJson::None;
    }

    switch (JsonVal->Type)
    {
    case EJson::None:
        return EGamebaseJson::None;

    case EJson::Null:
        return EGamebaseJson::Null;

    case EJson::String:
        return EGamebaseJson::String;

    case EJson::Number:
        return EGamebaseJson::Number;

    case EJson::Boolean:
        return EGamebaseJson::Boolean;

    case EJson::Array:
        return EGamebaseJson::Array;

    case EJson::Object:
        return EGamebaseJson::Object;

    default:
        return EGamebaseJson::None;
    }
}

FString UGamebaseJsonValue::GetTypeString() const
{
    if (!JsonVal.IsValid())
    {
        return "None";
    }

    switch (JsonVal->Type)
    {
    case EJson::None:
        return TEXT("None");

    case EJson::Null:
        return TEXT("Null");

    case EJson::String:
        return TEXT("String");

    case EJson::Number:
        return TEXT("Number");

    case EJson::Boolean:
        return TEXT("Boolean");

    case EJson::Array:
        return TEXT("Array");

    case EJson::Object:
        return TEXT("Object");

    default:
        return TEXT("None");
    }
}

bool UGamebaseJsonValue::IsNull() const
{
    if (!JsonVal.IsValid())
    {
        return true;
    }

    return JsonVal->IsNull();
}

float UGamebaseJsonValue::AsNumber() const
{
    if (!JsonVal.IsValid())
    {
        ErrorMessage(TEXT("Number"));
        return 0.f;
    }

    return JsonVal->AsNumber();
}

FString UGamebaseJsonValue::AsString() const
{
    if (!JsonVal.IsValid())
    {
        ErrorMessage(TEXT("String"));
        return FString();
    }

    return JsonVal->AsString();
}

bool UGamebaseJsonValue::AsBool() const
{
    if (!JsonVal.IsValid())
    {
        ErrorMessage(TEXT("Boolean"));
        return false;
    }

    return JsonVal->AsBool();
}

TArray<UGamebaseJsonValue*> UGamebaseJsonValue::AsArray() const
{
    TArray<UGamebaseJsonValue*> OutArray;

    if (!JsonVal.IsValid())
    {
        ErrorMessage(TEXT("Array"));
        return OutArray;
    }

    TArray< TSharedPtr<FJsonValue> > ValArray = JsonVal->AsArray();
    for (auto Value : ValArray)
    {
        UGamebaseJsonValue* NewValue = NewObject<UGamebaseJsonValue>();
        NewValue->SetRootValue(Value);

        OutArray.Add(NewValue);
    }

    return OutArray;
}

UGamebaseJsonObject* UGamebaseJsonValue::AsObject()
{
    if (!JsonVal.IsValid())
    {
        ErrorMessage(TEXT("Object"));
        return nullptr;
    }

    TSharedPtr<FJsonObject> NewObj = JsonVal->AsObject();

    UGamebaseJsonObject* JsonObj = NewObject<UGamebaseJsonObject>();
    JsonObj->SetRootObject(NewObj);

    return JsonObj;
}

//////////////////////////////////////////////////////////////////////////
// Helpers

void UGamebaseJsonValue::ErrorMessage(const FString& InType) const
{
    //GAMEBASE_LOG_ERROR("Json Value of type '%s' used as a '%s'.", *GetTypeString(), *InType);
}