#include "GamebaseAuthTypes.h"

#include "GamebaseAuthProviderCredential.h"
#include "GamebaseError.h"
#include "Utilities/GamebaseValueObjectSerializerMacros.h"

namespace GamebaseErrorExtras
{
    const FString BanInfoKey{ TEXT("ban") };
    const FString ForcingMappingTicket{ TEXT("forcingMappingTicket") };
    const FString FailTransferAccount{ TEXT("failTransferAccount") };
}


void FGamebaseGraceBanInfo::Serialize(FJsonSerializerBase& Serializer, bool bFlatObject)
{
    auto PaymentStatusSerialize = [](FPaymentStatus& ResultType, FJsonSerializerBase& Serializer, bool bFlatObject)
    {
        BEGIN_GAMEBASE_VO_SERIALIZER
        GAMEBASE_VO_SERIALIZE_IN("amount", Amount);
        GAMEBASE_VO_SERIALIZE_IN("count", Count);
        GAMEBASE_VO_SERIALIZE_IN("currency", Currency);
        END_GAMEBASE_VO_SERIALIZER
    };

    auto ReleaseRuleConditionSerialize = [](FReleaseRuleCondition& ResultType, FJsonSerializerBase& Serializer, bool bFlatObject)
    {
        BEGIN_GAMEBASE_VO_SERIALIZER
        GAMEBASE_VO_SERIALIZE_IN("amount", Amount);
        GAMEBASE_VO_SERIALIZE_IN("count", Count);
        GAMEBASE_VO_SERIALIZE_IN("currency", Currency);
        GAMEBASE_VO_SERIALIZE_IN("conditionType", ConditionType);
        END_GAMEBASE_VO_SERIALIZER
    };
    
    BEGIN_GAMEBASE_VO_SERIALIZER
    GAMEBASE_VO_SERIALIZE("gracePeriodDate", GracePeriodDate);
    GAMEBASE_VO_SERIALIZE("message", Message);
    GAMEBASE_VO_SERIALIZE_OPTIONAL_OBJECT_SERIALIZABLE_INNER("paymentStatus", PaymentStatus, PaymentStatusSerialize);
    GAMEBASE_VO_SERIALIZE_OPTIONAL_OBJECT_SERIALIZABLE_INNER("releaseRuleCondition", ReleaseRuleCondition, ReleaseRuleConditionSerialize);
    END_GAMEBASE_VO_SERIALIZER
}

void FGamebaseTemporaryWithdrawalInfo::Serialize(FJsonSerializerBase& Serializer, bool bFlatObject)
{
    BEGIN_GAMEBASE_VO_SERIALIZER
    GAMEBASE_VO_SERIALIZE("gracePeriodDate", GracePeriodDate);
    END_GAMEBASE_VO_SERIALIZER
}

void FGamebaseAuthMappingInfo::Serialize(FJsonSerializerBase& Serializer, bool bFlatObject)
{
    BEGIN_GAMEBASE_VO_SERIALIZER
    GAMEBASE_VO_SERIALIZE("authKey", AuthKey);
    GAMEBASE_VO_SERIALIZE("authSystem", AuthSystem);
    GAMEBASE_VO_SERIALIZE("idPCode", IdpCode);
    GAMEBASE_VO_SERIALIZE("regDate", RegDate);
    GAMEBASE_VO_SERIALIZE("userId", UserId);
    END_GAMEBASE_VO_SERIALIZER
}

void FGamebaseAuthToken::FToken::Serialize(
    FJsonSerializerBase& Serializer,
    const bool bFlatObject)
{
    BEGIN_GAMEBASE_VO_SERIALIZER
    GAMEBASE_VO_SERIALIZE("sourceIdPCode", SourceIdpCode);
    GAMEBASE_VO_SERIALIZE("accessToken", AccessToken);
    GAMEBASE_VO_SERIALIZE_OPTIONAL("accessTokenSecret", AccessTokenSecret);
    GAMEBASE_VO_SERIALIZE_OPTIONAL("subCode", SubCode);
    GAMEBASE_VO_SERIALIZE_OPTIONAL("expiresIn", ExpiresIn);
    GAMEBASE_VO_SERIALIZE_OPTIONAL_VARIANT_MAP("extraParams", ExtraParams);
    END_GAMEBASE_VO_SERIALIZER
}

FGamebaseVariantMap FGamebaseAuthToken::FToken::ToCredentialInfo() const
{
    FGamebaseVariantMap CredentialInfo;
    
    CredentialInfo.Add(GamebaseAuthProviderCredential::ProviderName, SourceIdpCode);
    CredentialInfo.Add(GamebaseAuthProviderCredential::GamebaseAccessToken, AccessToken);

    if (SubCode.IsSet())
    {
        CredentialInfo.Add(GamebaseAuthProviderCredential::SubCode, *SubCode);
    }
    
    if (ExtraParams.IsSet())
    {
        CredentialInfo.Add(GamebaseAuthProviderCredential::ExtraParams, *ExtraParams);
    }
    
    return CredentialInfo;
}

void FGamebaseAuthToken::FMember::Serialize(
    FJsonSerializerBase& Serializer,
    const bool bFlatObject)
{
    BEGIN_GAMEBASE_VO_SERIALIZER
    GAMEBASE_VO_SERIALIZE("appId", AppId);
    GAMEBASE_VO_SERIALIZE_ARRAY_SERIALIZABLE("authSystem", AuthList, FGamebaseAuthMappingInfo);
    GAMEBASE_VO_SERIALIZE("lastLoginDate", LastLoginDate);
    GAMEBASE_VO_SERIALIZE("regDate", RegDate);
    GAMEBASE_VO_SERIALIZE("userId", UserId);
    GAMEBASE_VO_SERIALIZE("valid", Valid);
    GAMEBASE_VO_SERIALIZE_OPTIONAL_OBJECT_SERIALIZABLE_INNER_VO("temporaryWithdrawal", TemporaryWithdrawal);
    GAMEBASE_VO_SERIALIZE_OPTIONAL_OBJECT_SERIALIZABLE_INNER_VO("graceBan", GraceBan);
    END_GAMEBASE_VO_SERIALIZER
}

void FGamebaseAuthToken::Serialize(
    FJsonSerializerBase& Serializer,
    const bool bFlatObject)
{
    BEGIN_GAMEBASE_VO_SERIALIZER
    GAMEBASE_VO_SERIALIZE_OBJECT_SERIALIZABLE("token", Token);
    GAMEBASE_VO_SERIALIZE_OBJECT_SERIALIZABLE("member", Member);
    END_GAMEBASE_VO_SERIALIZER
}

FGamebaseBanInfoPtr FGamebaseBanInfo::From(const FGamebaseError* Error)
{
    if (Error == nullptr)
    {
        return nullptr;
    }
    
    const TOptional<FString>& JsonString = Error->FindExtra(GamebaseErrorExtras::BanInfoKey);
    if (JsonString.IsSet() == false)
    {
        return nullptr;
    }

    const FGamebaseBanInfoPtr BanInfo = MakeShared<FGamebaseBanInfo, ESPMode::ThreadSafe>();
    BanInfo->FromJson(JsonString.GetValue());
    
    return BanInfo;
}

void FGamebaseBanInfo::Serialize(FJsonSerializerBase& Serializer, bool bFlatObject)
{
    BEGIN_GAMEBASE_VO_SERIALIZER
    GAMEBASE_VO_SERIALIZE("userId", UserId);
    GAMEBASE_VO_SERIALIZE("banType", BanType);
    GAMEBASE_VO_SERIALIZE("beginDate", BeginDate);
    GAMEBASE_VO_SERIALIZE("endDate", EndDate);
    GAMEBASE_VO_SERIALIZE("message", Message);
    GAMEBASE_VO_SERIALIZE("csInfo", CsInfo);
    GAMEBASE_VO_SERIALIZE("csUrl", CsUrl);
    END_GAMEBASE_VO_SERIALIZER
}

void FGamebaseTransferAccountInfo::Serialize(FJsonSerializerBase& Serializer, bool bFlatObject)
{
    auto AccountSerialize = [](FAccount& ResultType, FJsonSerializerBase& Serializer, bool bFlatObject)
    {
        BEGIN_GAMEBASE_VO_SERIALIZER
        GAMEBASE_VO_SERIALIZE_IN("id", Id);
        GAMEBASE_VO_SERIALIZE_IN("password", Password);
        END_GAMEBASE_VO_SERIALIZER
    };
    
    auto ConditionSerialize = [](FCondition& ResultType, FJsonSerializerBase& Serializer, bool bFlatObject)
    {
        BEGIN_GAMEBASE_VO_SERIALIZER
        GAMEBASE_VO_SERIALIZE_IN("transferAccountType", TransferAccountType);
        GAMEBASE_VO_SERIALIZE_IN("expirationType", ExpirationType);
        GAMEBASE_VO_SERIALIZE_IN("expirationDate", ExpirationDate);
        END_GAMEBASE_VO_SERIALIZER
    };
    
    BEGIN_GAMEBASE_VO_SERIALIZER
    GAMEBASE_VO_SERIALIZE("issuedType", IssuedType);
    GAMEBASE_VO_SERIALIZE_OBJECT_SERIALIZABLE_INNER("account", Account, AccountSerialize);
    GAMEBASE_VO_SERIALIZE_OBJECT_SERIALIZABLE_INNER("condition", Condition, ConditionSerialize);
    END_GAMEBASE_VO_SERIALIZER
}

FGamebaseTransferAccountFailInfoPtr FGamebaseTransferAccountFailInfo::From(const FGamebaseError* Error)
{
    if (Error == nullptr)
    {
        return nullptr;
    }
    
    const TOptional<FString>& JsonString = Error->FindExtra(GamebaseErrorExtras::FailTransferAccount);
    if (!JsonString.IsSet())
    {
        return nullptr;
    }
    
    const FGamebaseTransferAccountFailInfoPtr TransferAccountFailInfo = MakeShared<FGamebaseTransferAccountFailInfo, ESPMode::ThreadSafe>();
    TransferAccountFailInfo->FromJson(JsonString.GetValue());
    
    return TransferAccountFailInfo;
}

void FGamebaseTransferAccountFailInfo::Serialize(FJsonSerializerBase& Serializer, bool bFlatObject)
{
    BEGIN_GAMEBASE_VO_SERIALIZER
    GAMEBASE_VO_SERIALIZE("appId", AppId);
    GAMEBASE_VO_SERIALIZE("id", Id);
    GAMEBASE_VO_SERIALIZE("status", Status);
    GAMEBASE_VO_SERIALIZE("failCount", FailCount);
    GAMEBASE_VO_SERIALIZE("blockEndDate", BlockEndDate);
    GAMEBASE_VO_SERIALIZE("regDate", RegDate);
    END_GAMEBASE_VO_SERIALIZER
}

FGamebaseForcingMappingTicketPtr FGamebaseForcingMappingTicket::From(const FGamebaseError* Error)
{
    if (Error == nullptr)
    {
        return nullptr;
    }
    
    const TOptional<FString>& JsonString = Error->FindExtra(GamebaseErrorExtras::ForcingMappingTicket);
    if (!JsonString.IsSet())
    {
        return nullptr;
    }
    
    const FGamebaseForcingMappingTicketPtr ForcingMappingTicket = MakeShared<FGamebaseForcingMappingTicket, ESPMode::ThreadSafe>();
    ForcingMappingTicket->FromJson(JsonString.GetValue());
    
    return ForcingMappingTicket;
}

void FGamebaseForcingMappingTicket::Serialize(FJsonSerializerBase& Serializer, bool bFlatObject)
{
    BEGIN_GAMEBASE_VO_SERIALIZER
    GAMEBASE_VO_SERIALIZE("userId", UserId);
    GAMEBASE_VO_SERIALIZE("mappedUserId", MappedUserId);
    GAMEBASE_VO_SERIALIZE("mappedUserValid", MappedUserValid);
    GAMEBASE_VO_SERIALIZE("idPCode", IdpCode);
    GAMEBASE_VO_SERIALIZE("forcingMappingKey", ForcingMappingKey);
    GAMEBASE_VO_SERIALIZE("expirationDate", ExpirationDate);
    GAMEBASE_VO_SERIALIZE("accessToken", AccessToken);
    GAMEBASE_VO_SERIALIZE_OPTIONAL("subCode", SubCode);
    GAMEBASE_VO_SERIALIZE_OPTIONAL_VARIANT_MAP("extraParams", ExtraParams);
    END_GAMEBASE_VO_SERIALIZER
}