﻿#include "GamebaseWebBrowserManager.h"

#include "GamebaseDebugLogger.h"
#include "GamebaseErrorCode.h"
#include "GamebaseInternalResult.h"
#include "GamebaseWebBrowserConstants.h"
#include "GamebaseWebBrowserSchemeHandler.h"
#include "GamebaseWebBrowserStyleSet.h"
#include "GamebaseWebBrowserTypes.h"
#include "GamebaseWebBrowserUtils.h"
#include "GamebaseWebBrowserViewModel.h"
#include "IWebBrowserSingleton.h"
#include "SGamebaseWebBrowser.h"
#include "WebBrowserModule.h"
#include "Engine/GameViewportClient.h"

class FGamebaseWebBrowserManager final
    : public IGamebaseWebBrowserManager
{
    struct FBrowser
    {
        FName InstanceId;
        FGamebaseWebBrowserClose CloseDelegate;
        TArray<FGamebaseSchemeHandlerFactoryPtr> SchemeFactories;
        FGamebaseErrorResult Result;
        TSharedPtr<SGamebaseWebBrowser> Window;
        TWeakObjectPtr<UGameViewportClient> ViewportClient;
    };
    
public:
    virtual ~FGamebaseWebBrowserManager() override;
    
    virtual void Open(const FGamebaseWebBrowserParams& Params) override;
    virtual void Close() override;
    
private:
    bool IsPendingBrowser() const { return PendingBrowser.IsValid(); }

    void FinishBrowser();
    
    void RegisterBrowserSchemes(FBrowser* Params, const TArray<FString>& SchemeList);
    void UnregisterBrowserSchemes(FBrowser* Params);

private:
    TUniquePtr<FBrowser> PendingBrowser;
};

FGamebaseWebBrowserManager::~FGamebaseWebBrowserManager()
{
    Close();
}

void FGamebaseWebBrowserManager::Open(const FGamebaseWebBrowserParams& Params)
{
    if (IsPendingBrowser())
    {
        Params.CloseCallback.ExecuteIfBound(FGamebaseErrorResult(
            FGamebaseError(
                GamebaseErrorCode::WEBVIEW_UNKNOWN_ERROR,
                TEXT("A web browser is already in use."),
                GamebaseWebBrowser::Domain)));
        return;
    }
    
    if (Params.Url.IsEmpty())
    {
        Params.CloseCallback.ExecuteIfBound(FGamebaseErrorResult(
            FGamebaseError(
                GamebaseErrorCode::INVALID_PARAMETER,
                TEXT("URL is empty."),
                GamebaseWebBrowser::Domain)));
        return;
    }

    const auto CloseFunc = [this, CloseCallback = Params.CloseCallback]
    {
        if (!IsPendingBrowser())
        {
            return;
        }
        
        FinishBrowser();
    };
    
    const auto ErrorFunc = [this](const int32 Code, const FString& Message)
    {
        if (!IsPendingBrowser())
        {
            return;
        }

        PendingBrowser->Result = FGamebaseErrorResult(FGamebaseError(Code, Message, GamebaseWebBrowser::Domain));
        FinishBrowser();
    };
    
    const auto SchemeFunc = [this, SchemeCallback = Params.SchemeEventCallback](const FString& Url)
    {
        if (!IsPendingBrowser())
        {
            return;
        }

        SchemeCallback.ExecuteIfBound(Url);
    };
    
    TSharedRef<FGamebaseWebBrowserViewModel> ViewModel = FGamebaseWebBrowserViewModel::Create(
        Params.Url,
        Params.WidgetSettings.ToSharedRef(),
        Params.ContextSettings.ToSharedRef(),
        Params.SchemeList,
        CloseFunc,
        ErrorFunc,
        SchemeFunc,
        Params.bConsumeInput);
    
    PendingBrowser = MakeUnique<FBrowser>();
    
    RegisterBrowserSchemes(PendingBrowser.Get(), Params.SchemeList);
    
    PendingBrowser->InstanceId = *FGuid::NewGuid().ToString();
    PendingBrowser->ViewportClient = Params.ViewportClient;
    PendingBrowser->CloseDelegate = Params.CloseCallback;
    PendingBrowser->Window = SNew(SGamebaseWebBrowser, ViewModel).StyleSet(&FGamebaseWebBrowserStyleSet::Get());
    
    if (PendingBrowser->ViewportClient.IsValid())
    {
        PendingBrowser->ViewportClient->AddViewportWidgetContent(PendingBrowser->Window.ToSharedRef(), 10000000);
    }
}

void FGamebaseWebBrowserManager::Close()
{
    if (!PendingBrowser.IsValid())
    {
        return;
    }
    
    FinishBrowser();
}

void FGamebaseWebBrowserManager::FinishBrowser()
{
    if (PendingBrowser.IsValid())
    {
        if (PendingBrowser->ViewportClient.IsValid())
        {
            PendingBrowser->ViewportClient->RemoveViewportWidgetContent(PendingBrowser->Window.ToSharedRef());
        }
        
        PendingBrowser->CloseDelegate.ExecuteIfBound(PendingBrowser->Result);
        PendingBrowser.Reset();
    }
}

void FGamebaseWebBrowserManager::RegisterBrowserSchemes(FBrowser* Params, const TArray<FString>& SchemeList)
{
    IWebBrowserSingleton* WebBrowserSingleton = IWebBrowserModule::Get().GetSingleton();
    for (const auto& Url : SchemeList)
    {
        FGamebaseSchemeHandlerFactoryPtr SchemeHandler = MakeShared<FGamebaseSchemeHandlerFactory>();
        Params->SchemeFactories.Add(SchemeHandler);
        
        FString Scheme, Domain;
        GamebaseWebBrowserUtils::ParseScheme(Url, Scheme, Domain);
        if (!WebBrowserSingleton->RegisterSchemeHandlerFactory(Scheme, Domain, SchemeHandler.Get()))
        {
            GAMEBASE_LOG_GLOBAL_WARNING("RegisterSchemeHandlerFactory failed. (Scheme: %s, Domain: %s)", *Scheme, *Domain);
        }
    }
}

void FGamebaseWebBrowserManager::UnregisterBrowserSchemes(FBrowser* Params)
{
    IWebBrowserSingleton* WebBrowserSingleton = IWebBrowserModule::Get().GetSingleton();
    for (const auto& Scheme : Params->SchemeFactories)
    {
        if (!WebBrowserSingleton->UnregisterSchemeHandlerFactory(Scheme.Get()))
        {
            GAMEBASE_LOG_GLOBAL_WARNING("UnregisterSchemeHandlerFactory failed.");
        }
    }
}

TUniquePtr<IGamebaseWebBrowserManager> GamebaseWebBrowserManager::Create()
{
    return MakeUnique<FGamebaseWebBrowserManager>();
}
