#include "GamebaseStandalonePurchaseSteam.h"

#include "GamebaseDebugLogger.h"
#include "Constants/GamebaseStoreCode.h"
#include "Types/GamebasePurchaseConfiguration.h"

#if WITH_GAMEBASE_STEAMWORKS_SHARED
#pragma warning(push)
#pragma warning(disable : 4996)
#include "steam/isteamutils.h"
#pragma warning(pop)
#endif

namespace GamebasePurchaseSteam
{
    namespace IdPInfo
    {
        const FName StoreAppId(TEXT("StoreAppId"));
    }
    
    struct FPurchasableSteamData final : FGamebaseJsonSerializable
    {
        struct FProductInfo final : FGamebaseJsonSerializable
        {
            struct FPriceInfo final : FGamebaseJsonSerializable
            {
                float Price;
                FString Currency;
                FString Language;
                FString ProductName;
                
                BEGIN_GAMEBASE_JSON_SERIALIZER
                    GAMEBASE_JSON_SERIALIZE("price", Price);
                    GAMEBASE_JSON_SERIALIZE("currency", Currency);
                    GAMEBASE_JSON_SERIALIZE("language", Language);
                    GAMEBASE_JSON_SERIALIZE("productName", ProductName);
                END_GAMEBASE_JSON_SERIALIZER
            };
            
            int32 ProductNo;
            FString ProductId;
            FString ProductName;
            TArray<FPriceInfo> PriceList;
            
            BEGIN_GAMEBASE_JSON_SERIALIZER
                GAMEBASE_JSON_SERIALIZE("marketProductId", ProductId);
                GAMEBASE_JSON_SERIALIZE("productName", ProductName);
                GAMEBASE_JSON_SERIALIZE("productNo", ProductNo);
                GAMEBASE_JSON_SERIALIZE_ARRAY_SERIALIZABLE("priceList", PriceList, FPriceInfo);
            END_GAMEBASE_JSON_SERIALIZER
        };
        
        FString DefaultCurrency;
        TArray<FProductInfo> ProductList;
        
        BEGIN_GAMEBASE_JSON_SERIALIZER
            GAMEBASE_JSON_SERIALIZE("defaultCurrency", DefaultCurrency);
            GAMEBASE_JSON_SERIALIZE_ARRAY_SERIALIZABLE("productList", ProductList, FProductInfo);
        END_GAMEBASE_JSON_SERIALIZER
    };

    using FSteamPriceInfo = FPurchasableSteamData::FProductInfo::FPriceInfo;
    const FSteamPriceInfo* GetPriceInfo(const TArray<FSteamPriceInfo>& PriceList, const FString& Language, const FString& DefaultCurrency)
    {
        const FSteamPriceInfo* PriceInfo = PriceList.FindByPredicate([&Language](const FSteamPriceInfo& Info) {
            return Info.Language == Language;
        });

        if (!PriceInfo)
        {
            PriceInfo = PriceList.FindByPredicate([&DefaultCurrency](const FSteamPriceInfo& Info) {
                return Info.Currency == DefaultCurrency;
            });
        }

        if (!PriceInfo && PriceList.Num() > 0)
        {
            PriceInfo = &PriceList[0];
            GAMEBASE_LOG_GLOBAL_WARNING("No matching price information found. Using the first item in the list.");
        }

        return PriceInfo;
    }
}

void UGamebaseStandalonePurchaseSteam::PreInitialize(
    const FGamebasePurchaseConfiguration& Configuration)
{
    using namespace GamebasePurchaseSteam;
    
#if WITH_GAMEBASE_STEAMWORKS_SHARED
    if (SteamUtils())
    {
        const FString SteamworksAppId = FString::Printf(TEXT("%u"), SteamUtils()->GetAppID());
        if (!SteamworksAppId.Equals(Configuration.StoreAppId))
        {
            ErrorMessage = TEXT("The Steam AppID is different from console information.");
        }
    }
    else
    {
        ErrorMessage = TEXT("Steamworks initialization has not been completed.");
    }
#endif
}

FGamebaseIapConfiguration UGamebaseStandalonePurchaseSteam::GetConfiguration(const FGamebasePurchaseConfiguration& Configuration) const
{
    using namespace GamebasePurchaseSteam;
    
    FGamebaseIapConfiguration IapConfiguration;
    IapConfiguration.Type = GetUniqueKey();
#if WITH_GAMEBASE_STEAMWORKS_SHARED
    IapConfiguration.LibPath = FPaths::ConvertRelativePathToFull(FPaths::LaunchDir()) / TEXT("Engine/Binaries/ThirdParty/Steamworks") / STEAM_SDK_VER_PATH / TEXT("Win64/steam_api64.dll");
#endif
    IapConfiguration.LogLevel = 2;
    IapConfiguration.LogTarget = 3;
    IapConfiguration.Appkey = Configuration.IapAppkey;
    IapConfiguration.HostUrl = Configuration.ServerUrl;
    IapConfiguration.IdpInfo = {
        { IdPInfo::StoreAppId, Configuration.StoreAppId }
    };
    
    return IapConfiguration;
}

FIapPurchasableItemArray UGamebaseStandalonePurchaseSteam::ConvertItemListPurchasable(const FString& JsonString)
{
    using namespace GamebasePurchaseSteam;
    
    const FString DisplayLanguage = GetDisplayLanguage();
    
    FIapPurchasableItemArray Result;
    
    FPurchasableSteamData Response;
    Response.FromJson(JsonString);
    
    for (const FPurchasableSteamData::FProductInfo& Item : Response.ProductList)
    {
        FIapPurchasableItem ResultItem;
        ResultItem.ItemSeq = Item.ProductNo;
        ResultItem.MarketItemId = Item.ProductId;
        ResultItem.ProductType = TEXT("CONSUMABLE");
        ResultItem.bIsActive = true;

        if (Item.PriceList.Num() > 0)
        {
            const FSteamPriceInfo* PriceInfo = GetPriceInfo(Item.PriceList, DisplayLanguage, Response.DefaultCurrency);
            if (PriceInfo)
            {
                ResultItem.Price = PriceInfo->Price;
                ResultItem.Currency = PriceInfo->Currency;
                ResultItem.LocalizedPrice = FText::AsCurrency(PriceInfo->Price, PriceInfo->Currency).ToString();
                ResultItem.LocalizedTitle = PriceInfo->ProductName;
                ResultItem.LocalizedDescription = FString();
            }
            else
            {
                GAMEBASE_LOG_GLOBAL_WARNING("No price information available for the item. (MarketItemId: %s, DisplayLanguage: %s, DefaultCurrency: %s)",
                    *Item.ProductId, *DisplayLanguage, *Response.DefaultCurrency);
            }
        }
        else
        {
            GAMEBASE_LOG_GLOBAL_WARNING("The price information list for the item is empty. (MarketItemId: %s)", *Item.ProductId);
        }
        
        Result.Add(ResultItem);
    }
    
    return Result;
}

FString UGamebaseStandalonePurchaseSteam::GetUniqueKey() const
{
    return GamebaseStoreCode::Steam;
}

TOptional<FString> UGamebaseStandalonePurchaseSteam::GetStoreError() const
{
    return ErrorMessage;
}
