#include "GamebaseOnlineSubsystemAuthHandler.h"

#if WITH_GAMEBASE_EOS_OSS

#include "GamebaseDebugLogger.h"
#include "GamebaseErrorCode.h"
#include "Online.h"
#include "Auth/GamebaseSharedEOSDefines.h"

namespace GamebaseSharedEOS
{
    namespace OnlineSubsystemLoginType
    {
        const FString ExchangeCode(TEXT("exchangecode"));
        const FString Developer(TEXT("developer"));
        const FString AccountPortal(TEXT("accountportal"));
        const FString PersistentAuth(TEXT("persistentauth"));
    }

    constexpr int32 DefaultUserId = 0;
}

FGamebaseOnlineSubsystemAuthHandler::FGamebaseOnlineSubsystemAuthHandler(
    const TWeakObjectPtr<UGameInstance>& GameInstance)
    : GameInstance(GameInstance)
{
}

auto FGamebaseOnlineSubsystemAuthHandler::AsyncLoginWithExistingSession() -> FLoginAsyncFunc
{
    return [this]() -> FLoginResultFuture
    {
        const TSharedRef<FLoginResultPromise> Promise = MakeShared<FLoginResultPromise>();

        const IOnlineIdentityPtr Identity = Online::GetIdentityInterface(EOS_SUBSYSTEM);
        if (!Identity.IsValid())
        {
            Promise->SetValue(FGamebaseAuthEpicLoginResult(
                FGamebaseError(
                    GamebaseErrorCode::AUTH_EXTERNAL_LIBRARY_ERROR,
                    TEXT("OnlineIdentity interface not valid."),
                    GamebaseSharedEOS::Auth::Domain)));

            return Promise->GetFuture();
        }

        const FUniqueNetIdPtr NetId = Identity->GetUniquePlayerId(0);
        if (!NetId.IsValid())
        {
            Promise->SetValue(FGamebaseAuthEpicLoginResult(
                FGamebaseError(
                    GamebaseErrorCode::AUTH_EXTERNAL_LIBRARY_ERROR,
                    TEXT("UserId not valid."),
                    GamebaseSharedEOS::Auth::Domain)));

            return Promise->GetFuture();
        }

        if (Identity->GetLoginStatus(0) == ELoginStatus::LoggedIn)
        {
            GAMEBASE_LOG_GLOBAL_DEBUG("OSS EOS already logged in");

            const TSharedPtr<const FUserOnlineAccount> UserAccount = Identity->GetUserAccount(*NetId);
            if (!UserAccount.IsValid())
            {
                Promise->SetValue(FGamebaseAuthEpicLoginResult(
                    FGamebaseError(
                        GamebaseErrorCode::AUTH_EXTERNAL_LIBRARY_ERROR,
                        TEXT("Login was successful, but failed to retrieve AccessToken from OnlineSubsystem."),
                        GamebaseSharedEOS::Auth::Domain)));

                return Promise->GetFuture();
            }

            FGamebaseAuthEpicLoginData LoginData;
            LoginData.AccessToken = UserAccount->GetAccessToken();
            Promise->SetValue(FGamebaseAuthEpicLoginResult(LoginData));
            return Promise->GetFuture();
        }

        Promise->SetValue(FGamebaseAuthEpicLoginResult(
            FGamebaseError(
                GamebaseErrorCode::AUTH_EXTERNAL_LIBRARY_ERROR,
                TEXT("UserId not logined."),
                GamebaseSharedEOS::Auth::Domain)));

        return Promise->GetFuture();
    };
}

auto FGamebaseOnlineSubsystemAuthHandler::AsyncLoginWithExchangeCode() -> FLoginAsyncFunc
{
    return [this]() -> FLoginResultFuture
    {
        FString ExchangeCode;
        if (!FParse::Value(FCommandLine::Get(), TEXT("AUTH_PASSWORD="), ExchangeCode) || ExchangeCode.IsEmpty())
        {
            const TSharedRef<FLoginResultPromise> Promise = MakeShared<FLoginResultPromise>();
            Promise->SetValue(FGamebaseAuthEpicLoginResult(
                FGamebaseError(
                    GamebaseErrorCode::AUTH_EXTERNAL_LIBRARY_ERROR,
                    TEXT("ExchangeCode not found in command line"),
                    GamebaseSharedEOS::Auth::Domain)));
            return Promise->GetFuture();
        }
        
        FOnlineAccountCredentials Credentials;
        Credentials.Type = GamebaseSharedEOS::OnlineSubsystemLoginType::ExchangeCode;
        Credentials.Token = ExchangeCode;
        
        return LoginWithCredentialsAsync(Credentials);
    };
}

auto FGamebaseOnlineSubsystemAuthHandler::AsyncLoginWithPersistentAuth() -> FLoginAsyncFunc
{
    return [this]() -> FLoginResultFuture
    {
        FOnlineAccountCredentials Credentials;
        Credentials.Type = GamebaseSharedEOS::OnlineSubsystemLoginType::PersistentAuth;
        
        return LoginWithCredentialsAsync(Credentials);
    };
}

auto FGamebaseOnlineSubsystemAuthHandler::AsyncLoginWithAccountPortal() -> FLoginAsyncFunc
{
    return [this]() -> FLoginResultFuture
    {
        FOnlineAccountCredentials Credentials;
        Credentials.Type = GamebaseSharedEOS::OnlineSubsystemLoginType::AccountPortal;
        
        return LoginWithCredentialsAsync(Credentials);
    };
}

auto FGamebaseOnlineSubsystemAuthHandler::AsyncLogout() -> FLogoutAsyncFunc
{
    return [this]() -> FLogoutResultFuture
    {
        const TSharedRef<FLogoutResultPromise> Promise = MakeShared<FLogoutResultPromise>();
        
        const IOnlineIdentityPtr Identity = Online::GetIdentityInterface(EOS_SUBSYSTEM);
        const FUniqueNetIdPtr NetId = Identity->GetUniquePlayerId(GamebaseSharedEOS::DefaultUserId);
 
        if (NetId == nullptr || Identity->GetLoginStatus(GamebaseSharedEOS::DefaultUserId) != ELoginStatus::LoggedIn)
        {
            Promise->SetValue(FGamebaseAuthEpicLogoutResult(
                FGamebaseError(
                    GamebaseErrorCode::AUTH_EXTERNAL_LIBRARY_ERROR,
                    TEXT("OSS EOS not logged in"),
                    GamebaseSharedEOS::Auth::Domain)));
            return Promise->GetFuture();
        }
        
        TSharedPtr<FDelegateHandle> DelegateHandle = MakeShared<FDelegateHandle>();
        *DelegateHandle = Identity->AddOnLogoutCompleteDelegate_Handle(
            0,
            FOnLogoutCompleteDelegate::CreateLambda([DelegateHandle, Identity, Promise](
                const int32 LocalUserNum,
                const bool bWasSuccessful)
                {
                    if (bWasSuccessful)
                    {
                        constexpr FGamebaseAuthEpicLogoutData EpicLogoutData;
                        Promise->SetValue(FGamebaseAuthEpicLogoutResult(EpicLogoutData));
                    }
                    else
                    {
                        Promise->SetValue(FGamebaseAuthEpicLogoutResult(
                            FGamebaseError(
                                GamebaseErrorCode::AUTH_EXTERNAL_LIBRARY_ERROR,
                                TEXT("OSS EOS logout failed."),
                                GamebaseSharedEOS::Auth::Domain)));
                    }
                
                    Identity->ClearOnLogoutCompleteDelegate_Handle(LocalUserNum, *DelegateHandle);
                }));
    
        if (Identity->Logout(0) == false)
        {
            Promise->SetValue(FGamebaseAuthEpicLogoutResult(
                FGamebaseError(
                    GamebaseErrorCode::AUTH_EXTERNAL_LIBRARY_ERROR,
                    TEXT("OSS EOS logout failed."),
                    GamebaseSharedEOS::Auth::Domain)));
            
            Identity->ClearOnLogoutCompleteDelegate_Handle(0, *DelegateHandle);
        }

        return Promise->GetFuture();
    };
}

auto FGamebaseOnlineSubsystemAuthHandler::LoginWithCredentialsAsync(const FOnlineAccountCredentials& Credentials) -> FLoginResultFuture
{
    const TSharedRef<FLoginResultPromise> Promise = MakeShared<FLoginResultPromise>();
    
    const IOnlineIdentityPtr Identity = Online::GetIdentityInterface(EOS_SUBSYSTEM);
    if (!Identity.IsValid())
    {
        Promise->SetValue(FGamebaseAuthEpicLoginResult(
            FGamebaseError(
                GamebaseErrorCode::AUTH_EXTERNAL_LIBRARY_ERROR,
                TEXT("OnlineIdentity interface not valid."),
                GamebaseSharedEOS::Auth::Domain)));
        
        return Promise->GetFuture();
    }

    TSharedPtr<FDelegateHandle> DelegateHandle = MakeShared<FDelegateHandle>();
    *DelegateHandle = Identity->AddOnLoginCompleteDelegate_Handle(
        0,
        FOnLoginCompleteDelegate::CreateLambda([DelegateHandle, Identity, Promise](
            const int32 LocalUserNum,
            const bool bWasSuccessful,
            const FUniqueNetId& UserId,
            const FString& Error)
            {
                if (bWasSuccessful)
                {
                    const TSharedPtr<const FUserOnlineAccount> UserAccount = Identity->GetUserAccount(UserId);
                    if (UserAccount.IsValid())
                    {
                        FGamebaseAuthEpicLoginData LoginData;
                        LoginData.AccessToken = UserAccount->GetAccessToken();
                        Promise->SetValue(FGamebaseAuthEpicLoginResult(LoginData));
                    }
                    else
                    {
                        Promise->SetValue(FGamebaseAuthEpicLoginResult(
                            FGamebaseError(
                                GamebaseErrorCode::AUTH_EXTERNAL_LIBRARY_ERROR,
                                TEXT("Login was successful, but failed to retrieve AccessToken from OnlineSubsystem."),
                                GamebaseSharedEOS::Auth::Domain)));
                    }
                }
                else
                {
                    Promise->SetValue(FGamebaseAuthEpicLoginResult(
                        FGamebaseError(
                            GamebaseErrorCode::AUTH_EXTERNAL_LIBRARY_ERROR,
                            Error,
                            GamebaseSharedEOS::Auth::Domain)));
                }

                Identity->ClearOnLoginCompleteDelegate_Handle(LocalUserNum, *DelegateHandle);
            }));

    if (!Identity->Login(0, Credentials))
    {
        Promise->SetValue(FGamebaseAuthEpicLoginResult(
            FGamebaseError(
                GamebaseErrorCode::AUTH_EXTERNAL_LIBRARY_ERROR,
                TEXT("OSS EOS login failed."),
                GamebaseSharedEOS::Auth::Domain)));

        Identity->ClearOnLoginCompleteDelegate_Handle(0, *DelegateHandle);
        return Promise->GetFuture();
    }

    return Promise->GetFuture();
}

#endif // WITH_GAMEBASE_EOS_OSS
