#include "GamebaseEOSSDKAuthHandler.h"

#if WITH_GAMEBASE_EOS_NATIVE_SDK

#include "eos_auth.h"
#include "eos_sdk.h"
#include "GamebaseErrorCode.h"
#include "GamebaseSharedEOS.h"
#include "Auth/GamebaseSharedEOSDefines.h"

namespace GamebaseAuthEpic
{
    EOS_HPlatform GetPlatformHandle(const UGameInstance* GameInstance)
    {
        if (const UGamebaseSharedEOS* SharedEOS = GameInstance->GetSubsystem<UGamebaseSharedEOS>())
        {
            return SharedEOS->GetPlatformHandle();
        }

        return nullptr;
    }
    
    class FCallbackBase
    {
        static bool bShouldCancelAllCallbacks;

    public:
        virtual ~FCallbackBase() {}
        static bool ShouldCancelAllCallbacks() { return bShouldCancelAllCallbacks; }
        static void CancelAllCallbacks() { bShouldCancelAllCallbacks = true; }
    };
    
    bool FCallbackBase::bShouldCancelAllCallbacks = false;
    
    /** Class to handle all callbacks generically using a lambda to process callback results */
    template<typename CallbackFuncType, typename CallbackType>
    class TEOSCallback final :
        public FCallbackBase
    {
    public:
        TFunction<void(const CallbackType*)> CallbackLambda;

        TEOSCallback()
        {

        }
        virtual ~TEOSCallback() = default;
        
        CallbackFuncType GetCallbackPtr()
        {
            return &CallbackImpl;
        }

    private:
        static void EOS_CALL CallbackImpl(const CallbackType* Data)
        {
            if (EOS_EResult_IsOperationComplete(Data->ResultCode) == EOS_FALSE)
            {
                // Ignore
                return;
            }
            check(IsInGameThread());

            TEOSCallback* CallbackThis = static_cast<TEOSCallback*>(Data->ClientData);
            check(CallbackThis);

            if (ShouldCancelAllCallbacks())
            {
                delete CallbackThis;
                return;
            }

            check(CallbackThis->CallbackLambda);
            CallbackThis->CallbackLambda(Data);

            delete CallbackThis;
        }
    };
}

auto FGamebaseEOSSDKAuthHandler::AsyncLoginWithExistingSession() -> FLoginAsyncFunc
{
    return [this]() -> FLoginResultFuture
    {
        EOS_Auth_Credentials Credentials = {};
        Credentials.ApiVersion = EOS_AUTH_CREDENTIALS_API_LATEST;
        Credentials.Type = EOS_ELoginCredentialType::EOS_LCT_PersistentAuth;
        Credentials.Id = nullptr;
        Credentials.Token = nullptr;
    
        EOS_Auth_LoginOptions LoginOptions = {};
        LoginOptions.ApiVersion = EOS_AUTH_LOGIN_API_LATEST;
        LoginOptions.Credentials = &Credentials;
        
        return LoginWithCredentialsAsync(LoginOptions);
    };
}

auto FGamebaseEOSSDKAuthHandler::AsyncLoginWithExchangeCode() -> FLoginAsyncFunc
{
    return [this]() -> FLoginResultFuture
    {
        FString ExchangeCode;
        if (!FParse::Value(FCommandLine::Get(), TEXT("AUTH_PASSWORD="), ExchangeCode) || ExchangeCode.IsEmpty())
        {
            const TSharedRef<FLoginResultPromise> Promise = MakeShared<FLoginResultPromise>();
            Promise->SetValue(FGamebaseAuthEpicLoginResult(
                FGamebaseError(
                    GamebaseErrorCode::AUTH_EXTERNAL_LIBRARY_ERROR,
                    TEXT("ExchangeCode not found in command line"),
                    GamebaseSharedEOS::Auth::Domain)));
            return Promise->GetFuture();
        }
        
        EOS_Auth_Credentials Credentials = {};
        Credentials.ApiVersion = EOS_AUTH_CREDENTIALS_API_LATEST;
        Credentials.Type = EOS_ELoginCredentialType::EOS_LCT_ExchangeCode;
        Credentials.Token = TCHAR_TO_UTF8(*ExchangeCode);
        
        EOS_Auth_LoginOptions LoginOptions = {};
        LoginOptions.ApiVersion = EOS_AUTH_LOGIN_API_LATEST;
        LoginOptions.Credentials = &Credentials;
        
        return LoginWithCredentialsAsync(LoginOptions);
    };
}

auto FGamebaseEOSSDKAuthHandler::AsyncLoginWithPersistentAuth() -> FLoginAsyncFunc
{
    return [this]() -> FLoginResultFuture
    {
        EOS_Auth_Credentials Credentials = {};
        Credentials.ApiVersion = EOS_AUTH_CREDENTIALS_API_LATEST;
        Credentials.Type = EOS_ELoginCredentialType::EOS_LCT_PersistentAuth;
        Credentials.Id = nullptr;
        Credentials.Token = nullptr;
    
        EOS_Auth_LoginOptions LoginOptions = {};
        LoginOptions.ApiVersion = EOS_AUTH_LOGIN_API_LATEST;
        LoginOptions.Credentials = &Credentials;
        
        return LoginWithCredentialsAsync(LoginOptions);
    };
}

auto FGamebaseEOSSDKAuthHandler::AsyncLoginWithAccountPortal() -> FLoginAsyncFunc
{
    return [this]() -> FLoginResultFuture
    {
        EOS_Auth_Credentials Credentials;
        Credentials.ApiVersion = EOS_AUTH_CREDENTIALS_API_LATEST;
        Credentials.Type = EOS_ELoginCredentialType::EOS_LCT_AccountPortal;
    
        EOS_Auth_LoginOptions LoginOptions;
        LoginOptions.ApiVersion = EOS_AUTH_LOGIN_API_LATEST;
        LoginOptions.Credentials = &Credentials;
        
        return LoginWithCredentialsAsync(LoginOptions);
    };
}

auto FGamebaseEOSSDKAuthHandler::AsyncLogout() -> FLogoutAsyncFunc
{
    return [this]() -> FLogoutResultFuture
    {
        const TSharedRef<FLogoutResultPromise> Promise = MakeShared<FLogoutResultPromise>();
    
        const EOS_HAuth AuthHandle = EOS_Platform_GetAuthInterface(GamebaseAuthEpic::GetPlatformHandle(GameInstance.Get()));
        if (!AuthHandle)
        {
            Promise->SetValue(FGamebaseAuthEpicLogoutResult(
                FGamebaseError(
                    GamebaseErrorCode::AUTH_EXTERNAL_LIBRARY_ERROR,
                    TEXT("OnlineIdentity interface not valid."),
                    GamebaseSharedEOS::Auth::Domain)));
            
            return Promise->GetFuture();
        }

        if (EpicAccountId == nullptr)
        {
            Promise->SetValue(FGamebaseAuthEpicLogoutResult(
                FGamebaseError(
                    GamebaseErrorCode::AUTH_EXTERNAL_LIBRARY_ERROR,
                    TEXT("UserId not loggined."),
                    GamebaseSharedEOS::Auth::Domain)));
            
            return Promise->GetFuture();
        }
        
        using FLogoutCallback = GamebaseAuthEpic::TEOSCallback<EOS_Auth_OnLogoutCallback, EOS_Auth_LogoutCallbackInfo>;
    
        FLogoutCallback* CallbackObj = new FLogoutCallback();
        CallbackObj->CallbackLambda = [this, Promise, AuthHandle](const EOS_Auth_LogoutCallbackInfo* Data)
        {
            if (Data->ResultCode == EOS_EResult::EOS_Success)
            {
                EpicAccountId = Data->LocalUserId;
            
                EOS_Auth_Token* AuthToken = nullptr;
                EOS_Auth_CopyUserAuthTokenOptions Options;
                Options.ApiVersion = EOS_AUTH_COPYUSERAUTHTOKEN_API_LATEST;

                const EOS_EResult CopyResult = EOS_Auth_CopyUserAuthToken(AuthHandle, &Options, EpicAccountId, &AuthToken);
                if (CopyResult == EOS_EResult::EOS_Success)
                {
                    constexpr FGamebaseAuthEpicLogoutData EpicLogoutData;
                    Promise->SetValue(FGamebaseAuthEpicLogoutResult(EpicLogoutData));
                    
                    EOS_Auth_Token_Release(AuthToken);
                }
                else
                {
                    Promise->SetValue(FGamebaseAuthEpicLogoutResult(
                        FGamebaseError(
                            GamebaseErrorCode::AUTH_EXTERNAL_LIBRARY_ERROR,
                            TEXT("Login was successful, but failed to retrieve AccessToken from EOS SDK."),
                            GamebaseSharedEOS::Auth::Domain)));
                }
            }
            else
            {
                Promise->SetValue(FGamebaseAuthEpicLogoutResult(
                    FGamebaseError(
                        GamebaseErrorCode::AUTH_EXTERNAL_LIBRARY_ERROR,
                        *FString(EOS_EResult_ToString(Data->ResultCode)),
                        GamebaseSharedEOS::Auth::Domain)));
            }
        };

        EOS_Auth_LogoutOptions LogoutOptions;
        LogoutOptions.ApiVersion = EOS_AUTH_LOGOUT_API_LATEST;
        LogoutOptions.LocalUserId = EpicAccountId;
    
        EOS_Auth_Logout(AuthHandle, &LogoutOptions, CallbackObj, CallbackObj->GetCallbackPtr());
            
        return Promise->GetFuture();
    };
}

auto FGamebaseEOSSDKAuthHandler::LoginWithCredentialsAsync(const EOS_Auth_LoginOptions& LoginOptions) -> FLoginResultFuture
{
    const TSharedRef<FLoginResultPromise> Promise = MakeShared<FLoginResultPromise>();
    
    const EOS_HAuth AuthHandle = EOS_Platform_GetAuthInterface(GamebaseAuthEpic::GetPlatformHandle(GameInstance.Get()));
    if (!AuthHandle)
    {
        Promise->SetValue(FGamebaseAuthEpicLoginResult(
            FGamebaseError(
                GamebaseErrorCode::AUTH_EXTERNAL_LIBRARY_ERROR,
                TEXT("OnlineIdentity interface not valid."),
                GamebaseSharedEOS::Auth::Domain)));
        return Promise->GetFuture();
    }
    
    using FLoginCallback = GamebaseAuthEpic::TEOSCallback<EOS_Auth_OnLoginCallback, EOS_Auth_LoginCallbackInfo>;

    FLoginCallback* CallbackObj = new FLoginCallback();
    CallbackObj->CallbackLambda = [this, Promise, AuthHandle](const EOS_Auth_LoginCallbackInfo* Data)
    {
        if (Data->ResultCode == EOS_EResult::EOS_Success)
        {
            EpicAccountId = Data->LocalUserId;
            
            EOS_Auth_Token* AuthToken = nullptr;
            EOS_Auth_CopyUserAuthTokenOptions Options;
            Options.ApiVersion = EOS_AUTH_COPYUSERAUTHTOKEN_API_LATEST;

            const EOS_EResult CopyResult = EOS_Auth_CopyUserAuthToken(AuthHandle, &Options, EpicAccountId, &AuthToken);
            if (CopyResult == EOS_EResult::EOS_Success)
            {
                FGamebaseAuthEpicLoginData LoginData;
                LoginData.AccessToken = AuthToken->AccessToken;
                Promise->SetValue(FGamebaseAuthEpicLoginResult(LoginData));
                
                EOS_Auth_Token_Release(AuthToken);
            }
            else
            {
                Promise->SetValue(FGamebaseAuthEpicLoginResult(
                    FGamebaseError(
                        GamebaseErrorCode::AUTH_EXTERNAL_LIBRARY_ERROR,
                        TEXT("Login was successful, but failed to retrieve AccessToken from EOS SDK."),
                        GamebaseSharedEOS::Auth::Domain)));
            }
        }
        else
        {
            Promise->SetValue(FGamebaseAuthEpicLoginResult(
                FGamebaseError(
                    GamebaseErrorCode::AUTH_EXTERNAL_LIBRARY_ERROR,
                    *FString(EOS_EResult_ToString(Data->ResultCode)),
                    GamebaseSharedEOS::Auth::Domain)));
        }
    };

    EOS_Auth_Login(AuthHandle, &LoginOptions, CallbackObj, CallbackObj->GetCallbackPtr());
    
    return Promise->GetFuture();
}

#endif // WITH_GAMEBASE_EOS_NATIVE_SDK