﻿#pragma once

#include "CoreMinimal.h"
#include "GpLoggerTypes.h"

struct FGpLoggerLogData;

class IGpLoggerFilter
{
public:
    explicit IGpLoggerFilter(const FString& Name, const bool bIsEnabled)
        : Name(Name), bIsEnabled(bIsEnabled)
    {
    }
    virtual ~IGpLoggerFilter() = default;

    virtual bool IsFiltered(const FGpLoggerLogData& LogItem) = 0;
    virtual bool IsEnabled() const { return bIsEnabled; }
    virtual FString GetName() const { return Name; }
    
protected:
    const FString Name;
    const bool bIsEnabled;
};

class FGpLoggerNormalFilter final : public IGpLoggerFilter
{
public:
    explicit FGpLoggerNormalFilter(const bool bIsEnable)
        : IGpLoggerFilter(TEXT("NormalFilter"), bIsEnable)
    {
    }
    
    virtual bool IsFiltered(const FGpLoggerLogData& LogItem) override;
};

class FGpLoggerSessionFilter final : public IGpLoggerFilter
{
public:
    explicit FGpLoggerSessionFilter(const bool bIsEnable)
        : IGpLoggerFilter(TEXT("SessionFilter"), bIsEnable)    
    {
    }

    virtual bool IsFiltered(const FGpLoggerLogData& LogItem) override;
};

class FGpLoggerCrashFilter final : public IGpLoggerFilter
{
public:
    explicit FGpLoggerCrashFilter(const bool bIsEnable)
        : IGpLoggerFilter(TEXT("CrashFilter"), bIsEnable)    
    {
    }
    
    virtual bool IsFiltered(const FGpLoggerLogData& LogItem) override;
};

class FGpLoggerLogLevelFilter final : public IGpLoggerFilter
{
public:
    FGpLoggerLogLevelFilter(const bool bIsEnable, const EGpLogLevel LogLevel)
        : IGpLoggerFilter(TEXT("LogLevelFilter"), bIsEnable)
        , LogLevel(LogLevel)
    {
    }
    
    virtual bool IsFiltered(const FGpLoggerLogData& LogItem) override;

private:
    const EGpLogLevel LogLevel;
};

class FGpLoggerLogTypeFilter final : public IGpLoggerFilter
{
public:
    FGpLoggerLogTypeFilter(const bool bIsEnable, const TArray<FString>& LogType)
        : IGpLoggerFilter(TEXT("LogTypeFilter"), bIsEnable)
        , LogType(LogType)
    {
    }
    
    virtual bool IsFiltered(const FGpLoggerLogData& LogItem) override;

protected:
    const TArray<FString> LogType;
};

class FGpLoggerDuplicateFilter final : public IGpLoggerFilter
{
    static constexpr int64 DefaultDuplicateFilterExpireTimeMS = 2000;
    
public:
    FGpLoggerDuplicateFilter(const bool bIsEnable, const int32 ExpireTime)
        : IGpLoggerFilter(TEXT("DuplicateFilter"), bIsEnable)
        , ExpireTime(ExpireTime)
    {
    }
    
    virtual bool IsFiltered(const FGpLoggerLogData& LogItem) override;

private:
	mutable FCriticalSection CriticalSection;
    
    int32 ExpireTime;
    TMap<uint32, int64> DuplicateInfoMap;
};
