﻿#include "GpLoggerBackupStorage.h"

#include "GpLoggerLogInfo.h"
#include "Kismet/GameplayStatics.h"
#include "Misc/Base64.h"
#include "Misc/Paths.h"
#include "Misc/FileHelper.h"

namespace GpFileSecure
{
    FString HashProjectKey(const FString& ProjectKey)
    {
        FSHAHash Hash;
        FSHA1::HashBuffer(TCHAR_TO_ANSI(*ProjectKey), ProjectKey.Len(), Hash.Hash);
        return Hash.ToString();
    }
}


FGpBulkLogDataPtr FGpLoggerBackupStorage::LoadData(const FGpLoggerServiceData& ServiceData)
{
    const FString HashedAppkey = GpFileSecure::HashProjectKey(ServiceData.Appkey);

    if (BackupLogs.Contains(HashedAppkey))
    {
        const auto SaveData = BackupLogs.FindRef(HashedAppkey)->BulkLogs[0];
        return MakeShared<FGpLoggerBulkLogData, ESPMode::ThreadSafe>(ServiceData, SaveData.Logs, SaveData.Uuid);
    }
    
    const FString SlotName = FString::Printf(TEXT("GpLogger_%s"), *HashedAppkey);
    if (!UGameplayStatics::DoesSaveGameExist(SlotName, 0))
    {
        return nullptr;
    }
    
    UGpLoggerLogSave* SaveLogObject = Cast<UGpLoggerLogSave>(UGameplayStatics::LoadGameFromSlot(SlotName, 0));

    if (SaveLogObject)
    {
        BackupLogs.FindOrAdd(HashedAppkey, SaveLogObject);
        SaveLogObject->AddToRoot();

        if (SaveLogObject->BulkLogs.Num() > 0)
        {
            const auto SaveData = BackupLogs.FindRef(HashedAppkey)->BulkLogs[0];
            return MakeShared<FGpLoggerBulkLogData, ESPMode::ThreadSafe>(ServiceData, SaveData.Logs, SaveData.Uuid);
        }
    }

    return nullptr;
}

bool FGpLoggerBackupStorage::SaveData(const FGpLoggerServiceData& ServiceData, const FGpLoggerBulkLogData& Info)
{
    FString HashedAppkey = GpFileSecure::HashProjectKey(ServiceData.Appkey);

    UGpLoggerLogSave* SaveLogObject = BackupLogs.FindRef(HashedAppkey);
    if (SaveLogObject)
    {
        for (int i = 0; i < SaveLogObject->BulkLogs.Num(); i++)
        {
            const auto SaveLog = SaveLogObject->BulkLogs[i];
            if (SaveLog.Uuid == Info.GetUuid())
            {
                return true;
            }
        }
        
        FGpSaveBulkData SaveData;
        SaveData.Uuid = FGuid::NewGuid().ToString();
        SaveData.Logs.Append(Info.GetLogs());

        SaveLogObject->BulkLogs.Add(SaveData);
    }
    else
    {
        SaveLogObject = Cast<UGpLoggerLogSave>(UGameplayStatics::CreateSaveGameObject(UGpLoggerLogSave::StaticClass()));

        SaveLogObject->Appkey = HashedAppkey;
        
        FGpSaveBulkData SaveData;
        SaveData.Uuid = FGuid::NewGuid().ToString();
        SaveData.Logs.Append(Info.GetLogs());

        SaveLogObject->BulkLogs.Add(SaveData);

        SaveLogObject->AddToRoot();

        BackupLogs.FindOrAdd(HashedAppkey, SaveLogObject);
    }

    return UGameplayStatics::SaveGameToSlot(SaveLogObject, FString::Printf(TEXT("GpLogger_%s"), *HashedAppkey), 0);
}

bool FGpLoggerBackupStorage::DeleteData(const FGpLoggerServiceData& ServiceData, const FGpLoggerBulkLogData& Info)
{
    const FString HashedAppkey = GpFileSecure::HashProjectKey(ServiceData.Appkey);

    if (Info.GetUuid().IsEmpty())
    {
        return true;
    }

    UGpLoggerLogSave* SaveLogObject = BackupLogs.FindRef(HashedAppkey);
    if (SaveLogObject == nullptr)
    {
        return true;
    }

    for (int i = 0; i < SaveLogObject->BulkLogs.Num(); i++)
    {
        const auto SaveLog = SaveLogObject->BulkLogs[i];
        if (SaveLog.Uuid == Info.GetUuid())
        {
            SaveLogObject->BulkLogs.RemoveAt(i);
            break;
        }
    }

    if (SaveLogObject->BulkLogs.Num() == 0)
    {
        BackupLogs.Remove(HashedAppkey);
        return UGameplayStatics::DeleteGameInSlot(FString::Printf(TEXT("GpLogger_%s"), *HashedAppkey), 0);
    }

    return UGameplayStatics::SaveGameToSlot(SaveLogObject, FString::Printf(TEXT("GpLogger_%s"), *HashedAppkey), 0);
}

void FGpLoggerBackupStorage::DeleteOldData(const FGpLoggerServiceData& ServiceData)
{
    static constexpr int64 RetentionPeriod = 2592000;      // 30 days
}
