#pragma once

#include "GpLoggerError.h"
#include "Misc/TVariant.h"

template <typename SuccessType, typename ErrorType>
class TGpLoggerResult
{
public:
    explicit TGpLoggerResult(const SuccessType& OkValue)
        : Storage(TInPlaceType<SuccessType>(), OkValue)
    {
    }

    explicit TGpLoggerResult(SuccessType&& OkValue)
        : Storage(TInPlaceType<SuccessType>(), MoveTemp(OkValue))
    {
    }

    explicit TGpLoggerResult(const ErrorType& ErrValue)
        : Storage(TInPlaceType<ErrorType>(), ErrValue)
    {
    }

    explicit TGpLoggerResult(ErrorType&& ErrValue)
        : Storage(TInPlaceType<ErrorType>(), MoveTemp(ErrValue))
    {
    }
    
    TGpLoggerResult(const TGpLoggerResult& Other) = default;
    TGpLoggerResult(TGpLoggerResult&& Other) = default;

    TGpLoggerResult& operator=(const TGpLoggerResult& Other)
    {
        if (&Other != this)
        {
            Storage = Other.Storage;
        }
        return *this;
    }

    TGpLoggerResult& operator=(TGpLoggerResult&& Other)
    {
        if (&Other != this)
        {
            Storage = MoveTemp(Other.Storage);
        }
        return *this;
    }

    virtual ~TGpLoggerResult() = default;

    bool IsOk() const
    {
        return Storage.template IsType<SuccessType>();
    }

    bool IsError() const
    {
        return Storage.template IsType<ErrorType>();
    }

    const SuccessType& GetOkValue() const
    {
        checkf(IsOk(), TEXT("It is an error to call GetOkValue() on a TResult that does not hold an ok value. Please either check IsOk() or use TryGetOkValue"));
        return Storage.template Get<SuccessType>();
    }

    SuccessType& GetOkValue()
    {
        checkf(IsOk(), TEXT("It is an error to call GetOkValue() on a TResult that does not hold an ok value. Please either check IsOk() or use TryGetOkValue"));
        return Storage.template Get<SuccessType>();
    }

    const ErrorType& GetErrorValue() const
    {
        checkf(IsError(), TEXT("It is an error to call GetErrorValue() on a TResult that does not hold an error value. Please either check IsError() or use TryGetErrorValue"));
        return Storage.template Get<ErrorType>();
    }

    ErrorType& GetErrorValue()
    {
        checkf(IsError(), TEXT("It is an error to call GetErrorValue() on a TResult that does not hold an error value. Please either check IsError() or use TryGetErrorValue"));
        return Storage.template Get<ErrorType>();
    }

    const SuccessType* TryGetOkValue() const
    {
        return const_cast<TGpLoggerResult*>(this)->TryGetOkValue();
    }

    SuccessType* TryGetOkValue()
    {
        return Storage.template TryGet<SuccessType>();
    }

    const ErrorType* TryGetErrorValue() const
    {
        return const_cast<TGpLoggerResult*>(this)->TryGetErrorValue();
    }

    ErrorType* TryGetErrorValue()
    {
        return Storage.template TryGet<ErrorType>();
    }
    
    const SuccessType& GetOkOrDefault(const SuccessType& DefaultValue) const
    {
        return IsOk() ? GetOkValue() : DefaultValue;
    }

protected:
    TGpLoggerResult() = default;

private:
    TVariant<SuccessType, ErrorType> Storage;
};

template <typename OpType>
class TGpLoggerInternalResult final : public TGpLoggerResult<OpType, FGpLoggerError>
{
public:
    using TGpLoggerResult<OpType, FGpLoggerError>::TGpLoggerResult;
};



template <typename ErrorType>
class TGpLoggerErrorResult
{
public:
    explicit TGpLoggerErrorResult()
        : Error()
    {
    }
    
    explicit TGpLoggerErrorResult(const ErrorType& ErrValue)
        : Error(ErrValue)
    {
    }
    
    explicit TGpLoggerErrorResult(ErrorType&& ErrValue)
        : Error(MoveTemp(ErrValue))
    {
    }
    
    TGpLoggerErrorResult(const TGpLoggerErrorResult& Other) = default;
    TGpLoggerErrorResult(TGpLoggerErrorResult&& Other) = default;

    TGpLoggerErrorResult& operator=(const TGpLoggerErrorResult& Other)
    {
        if (&Other != this)
        {
            Error = Other.Error;
        }
        return *this;
    }
    
    TGpLoggerErrorResult& operator=(TGpLoggerErrorResult&& Other)
    {
        if (&Other != this)
        {
            Error = MoveTemp(Other.Error);
        }
        return *this;
    }

    virtual ~TGpLoggerErrorResult() = default;

    bool IsOk() const
    {
        return Error.IsSet() == false;
    }
    
    bool IsError() const
    {
        return Error.IsSet();
    }
    
    const ErrorType& GetErrorValue() const
    {
        checkf(IsError(), TEXT("It is an error to call GetErrorValue() on a TResult that does not hold an error value. Please either check IsError() or use TryGetErrorValue"));
        return Error.GetValue();
    }

    ErrorType& GetErrorValue()
    {
        checkf(IsError(), TEXT("It is an error to call GetErrorValue() on a TResult that does not hold an error value. Please either check IsError() or use TryGetErrorValue"));
        return Error.GetValue();
    }

    const ErrorType* TryGetErrorValue() const
    {
        return const_cast<TGpLoggerErrorResult*>(this)->TryGetErrorValue();
    }
    
    ErrorType* TryGetErrorValue()
    {
        return Error.IsSet() ? &Error.GetValue() : nullptr;
    }

private:
    TOptional<ErrorType> Error;
};

class FGpLoggerErrorResult final : public TGpLoggerErrorResult<FGpLoggerError>
{
public:
    using TGpLoggerErrorResult::TGpLoggerErrorResult;
};

template <typename T>
FString ToLogString(const TGpLoggerInternalResult<T>& Result)
{
    if (Result.IsOk())
    {
        return ToLogString(Result.GetOkValue());
    }

    return ToLogString(Result.GetErrorValue());
}