#include "GpLoggerCoreSubsystem.h"

#include "GpBaseLogger.h"
#include "GpInternalInstanceLogger.h"
#include "GpLoggerEngineSubsystem.h"

TMap<FString, TWeakObjectPtr<UGpLoggerCoreSubsystem>> UGpLoggerCoreSubsystem::SubsystemMap;

UGpLoggerCoreSubsystem* UGpLoggerCoreSubsystem::GetSubsystem(const FString& InstanceName)
{
    const TWeakObjectPtr<UGpLoggerCoreSubsystem>* InstancePtr = SubsystemMap.Find(InstanceName);
    if (InstancePtr && InstancePtr->IsValid())
    {
        return InstancePtr->Get();
    }
    
    return nullptr;
}

int32 UGpLoggerCoreSubsystem::GetSubsystemCount()
{
    return SubsystemMap.Num();
}

void UGpLoggerCoreSubsystem::Initialize(const FGpLoggerConfiguration& Configuration)
{
    UGpBaseLogger::FInitializeParams Params;
    Params.Appkey = Configuration.Appkey;
    Params.ServiceZone = Configuration.ServiceZone;
    Params.SettingType = EGpLoggerSettingType::Console;
    Params.bIsCrashReporter = Configuration.bEnableCrashReporter;
    Logger->Initialize(Params);
    
    if (UGpLoggerEngineSubsystem* EngineSubsystem = GEngine->GetEngineSubsystem<UGpLoggerEngineSubsystem>())
    {
        UGpLoggerEngineSubsystem::FCrashInitializeParams CrashParams;
        CrashParams.bEnableReporter = Configuration.bEnableCrashReporter;
        CrashParams.bEnableErrorLog = false;
        EngineSubsystem->InitializeCrash(CrashParams);

        CrashDelegateHandle = EngineSubsystem->GetCrashDelegate().AddUObject(this, &UGpLoggerCoreSubsystem::OnCrashed);
    }
}

void UGpLoggerCoreSubsystem::Debug(const FString& Message, const TMap<FString, FString>& UserFields)
{
    Logger->Log(EGpLogLevel::Debug, Message, UserFields);
}

void UGpLoggerCoreSubsystem::Info(const FString& Message, const TMap<FString, FString>& UserFields)
{
    Logger->Log(EGpLogLevel::Info, Message, UserFields);
}

void UGpLoggerCoreSubsystem::Warn(const FString& Message, const TMap<FString, FString>& UserFields)
{
    Logger->Log(EGpLogLevel::Warn, Message, UserFields);
}

void UGpLoggerCoreSubsystem::Error(const FString& Message, const TMap<FString, FString>& UserFields)
{
    Logger->Log(EGpLogLevel::Error, Message, UserFields);
}

void UGpLoggerCoreSubsystem::Fatal(const FString& Message, const TMap<FString, FString>& UserFields)
{
    Logger->Log(EGpLogLevel::Fatal, Message, UserFields);
}

void UGpLoggerCoreSubsystem::SetUserField(const FString& Key, const FString& Value)
{
    Logger->SetUserField(Key, Value);
}

void UGpLoggerCoreSubsystem::SetLoggerListener(const FGpLoggerEventListener& Listener)
{
    Logger->SetListener(Listener);
}

void UGpLoggerCoreSubsystem::SetCrashListener(const FGpLoggerCrashListener& Listener)
{
    IsCrashFilterDelegate = Listener.OnCrashFilterDelegate;
}

void UGpLoggerCoreSubsystem::SetGameUserId(const FString& UserId)
{
    using namespace GpLogger;

    Logger->SetReserveField(LogField::UserID, UserId);

    for (TWeakObjectPtr<UGpInternalInstanceLogger> WeakInstance : InstanceLoggers)
    {
        if (WeakInstance.IsValid())
        {
            WeakInstance->SetReserveField(LogField::UserID, UserId);
        }
    }
}

UGpLoggerInstance* UGpLoggerCoreSubsystem::NewInstance(const FString& Appkey, const EGpLoggerServiceZone ServiceZone)
{
    UGpInternalInstanceLogger* Instance = NewObject<UGpInternalInstanceLogger>(this);
    Instance->Initialize(Appkey, ServiceZone, GetInstanceLoggerClass());
    InstanceLoggers.Emplace(Instance);

    return Instance;
}

void UGpLoggerCoreSubsystem::OnCrashed(const FGpLoggerCrashInternalData& CrashData)
{
    if (IsCrashFilter(CrashData))
    {
        return;
    }
    
    Logger->Reporter(CrashData);
}

bool UGpLoggerCoreSubsystem::IsCrashFilter(const FGpLoggerCrashInternalData& CrashData)
{
    if (IsCrashFilterDelegate.IsBound() == false)
    {
        return false;
    }

    FGpLoggerCrashData LogData;
    LogData.LogLevel = CrashData.LogLevel;
    LogData.Condition = CrashData.StackTrace;
    return IsCrashFilterDelegate.Execute(LogData);
}

void UGpLoggerCoreSubsystem::Initialize(FSubsystemCollectionBase& Collection)
{
    Super::Initialize(Collection);
    
    const FString InstanceName = GetGameInstance()->GetName();
    SubsystemMap.Emplace(InstanceName, this);
    
    Logger = NewObject<UGpBaseLogger>(this, GetLoggerClass());
}

void UGpLoggerCoreSubsystem::Deinitialize()
{
    Super::Deinitialize();
    
    InstanceLoggers.Empty();
    
    if (const UGpLoggerEngineSubsystem* EngineSubsystem = GEngine->GetEngineSubsystem<UGpLoggerEngineSubsystem>())
    {
        EngineSubsystem->GetCrashDelegate().Remove(CrashDelegateHandle);
    }
    
    const FString InstanceName = GetGameInstance()->GetName();
    SubsystemMap.Remove(InstanceName);
}
