﻿#include "GpLoggerCrashManager.h"

#include "GpLoggerInternalCrashTypes.h"
#include "GpLoggerDefines.h"
#include "GpLoggerInternalTypes.h"
#include "GpLoggerTypes.h"

namespace GpLoggerCrashHelper
{
    FString Base64Encode(const FString& Source)
    {
        TArray<uint8> ByteArray;
        const FTCHARToUTF8 StringSrc = FTCHARToUTF8(Source.GetCharArray().GetData());
        ByteArray.Append((uint8*)StringSrc.Get(), StringSrc.Length());

        return FBase64::Encode(ByteArray);
    }
    
    bool ReadMinidumpToBuffer(const FString& FilePath, TArray<uint8>& Buffer)
    {
        if (FFileHelper::LoadFileToArray(Buffer, *FilePath))
        {
            return true;
        }

        UE_LOG(LogTemp, Error, TEXT("Failed to read the file: %s"), *FilePath);
        return false;
    }
    
    FString ExtractCrashGUIDRoot(const FString& CrashConfigFilePath)
    {
        // 경로에서 폴더 부분을 추출
        FString FolderPath = FPaths::GetPath(CrashConfigFilePath);
    
        // 폴더 이름을 추출하여 GUID 부분만 얻음
        FString CrashGUIDRoot = FPaths::GetCleanFilename(FolderPath);
    
        return CrashGUIDRoot;
    }
    
    FString GetLatestCrashFolder(const FString& CrashGUIDRoot)
    {
        const FString GUIDPattern = CrashGUIDRoot + TEXT("_");
        
        const FString BaseCrashFolder = FPaths::Combine(FPaths::ProjectSavedDir(), TEXT("Crashes"));
        const FString FullCrashFolder = FPaths::ConvertRelativePathToFull(BaseCrashFolder);
        
        UE_LOG(LogTemp, Log, TEXT("Relative Path: %s"), *BaseCrashFolder);
        UE_LOG(LogTemp, Log, TEXT("Absolute Path: %s"), *FullCrashFolder);

        IPlatformFile& PlatformFile = FPlatformFileManager::Get().GetPlatformFile();
        int32 MaxIndex = -1;
        FString LatestCrashFolder;

        PlatformFile.IterateDirectory(*BaseCrashFolder, [&LatestCrashFolder, &GUIDPattern, &MaxIndex](const TCHAR* FileOrDirectory, const bool bIsDirectory) {
            if (bIsDirectory)
            {
                const FString FolderName = FPaths::GetCleanFilename(FileOrDirectory);
                UE_LOG(LogTemp, Log, TEXT("FolderName: %s"), *FolderName);
                
                if (FolderName.StartsWith(GUIDPattern))
                {
                    UE_LOG(LogTemp, Log, TEXT("GUIDPattern: %s"), *GUIDPattern);
                    
                    int32 UnderscoreIndex;
                    if (FolderName.FindLastChar(TEXT('_'), UnderscoreIndex))
                    {
                        UE_LOG(LogTemp, Log, TEXT("UnderscoreIndex: %d"), UnderscoreIndex);
                        
                        const FString IndexStr = FolderName.Mid(UnderscoreIndex + 1);
                        const int32 CurrentIndex = FCString::Atoi(*IndexStr);
                        
                        if (CurrentIndex > MaxIndex)
                        {
                            MaxIndex = CurrentIndex;
                            LatestCrashFolder = FileOrDirectory;
                        }
                    }
                }
            }
            return true;
        });

        UE_LOG(LogTemp, Log, TEXT("LatestCrashFolder: %s"), *LatestCrashFolder);
        return LatestCrashFolder;
    }
}

class FGpLoggerOutputDeviceError final : public FOutputDeviceError
{
public:
    FGpLoggerOutputDeviceError() = default;
    virtual ~FGpLoggerOutputDeviceError() override = default;
    
    virtual void Serialize(const TCHAR* V, const ELogVerbosity::Type Verbosity, const FName& Category) override
    {
        if (GIsGuarded)
        {
            switch (Verbosity)
            {
                case ELogVerbosity::Fatal:
                    OnSerializeFatal.ExecuteIfBound(FString(V), EGpLogLevel::Fatal, Category, {});
                break;
                default:  break;
            }
        }
        else
        {
            HandleError();
        }
    }
    
    virtual void Serialize(const TCHAR* V, const ELogVerbosity::Type Verbosity, const FName& Category, const double Time) override
    {
        Serialize(V, Verbosity, Category);
    }

    virtual void HandleError() override
    {
        // make sure we don't report errors twice
        static int32 CallCount = 0;
        int32 NewCallCount = FPlatformAtomics::InterlockedIncrement(&CallCount);
        if (NewCallCount != 1)
        {
            UE_LOG(LogGpLogger, Error, TEXT("HandleError re-entered.") );
            return;
        }
        
        const FString CrashHist(GErrorHist);
        OnSerializeFatal.ExecuteIfBound(CrashHist, EGpLogLevel::Fatal, FName(), {});
    }

    DECLARE_DELEGATE_FourParams(FOnSerializeMessage, const FString&, EGpLogLevel, const FName&, TOptional<double>);
    FOnSerializeMessage OnSerializeFatal;
};

class FGpLoggerOutputDevice final : public FOutputDevice
{
public:
    FGpLoggerOutputDevice()
    {
        GLog->AddOutputDevice(this);
    }
    
    virtual ~FGpLoggerOutputDevice() override
    {
        if (GLog != nullptr)
        {
            GLog->RemoveOutputDevice(this);
        }
    }
    
    virtual void Serialize(const TCHAR* V, const ELogVerbosity::Type Verbosity, const FName& Category) override
    {
        switch (Verbosity)
        {
            case ELogVerbosity::Error:
                {
                    OnSerializeError.ExecuteIfBound(FString(V), EGpLogLevel::Error, Category, {});
                    break;
                }
            case ELogVerbosity::Fatal:      // FGpLoggerOutputDeviceError에서 처리
            default:
                {
                    break;
                }
        }
    }
    virtual void Serialize(const TCHAR* V, const ELogVerbosity::Type Verbosity, const FName& Category, const double Time) override
    {
        Serialize(V, Verbosity, Category);
    }

    DECLARE_DELEGATE_FourParams(FOnSerializeMessage, const FString&, EGpLogLevel, const FName&, TOptional<double>);
    FOnSerializeMessage OnSerializeError;
};

FGpLoggerCrashManager::FGpLoggerCrashManager()
{
}

FGpLoggerCrashManager::~FGpLoggerCrashManager()
{
}

void FGpLoggerCrashManager::Initialize(const bool bEnableCrashReporter, const bool bEnableErrorReporter)
{
    if (bEnableCrashReporter == false)
    {
        GPLOGGER_LOG_DEBUG("Disable crash reporter");
        return;
    }

    if (OutputDeviceError.IsValid())
    {
        GPLOGGER_LOG_DEBUG("Already crash initialized");
        return;
    }

    OutputDeviceError = MakeUnique<FGpLoggerOutputDeviceError>();
    OutputDeviceError->OnSerializeFatal.BindRaw(this, &FGpLoggerCrashManager::OnCrashEvent);
    
    if (bEnableErrorReporter)
    {
        OutputDevice = MakeUnique<FGpLoggerOutputDevice>();
        OutputDevice->OnSerializeError.BindRaw(this, &FGpLoggerCrashManager::OnCrashEvent);
    }
    
    GError = OutputDeviceError.Get();
    FCoreDelegates::OnHandleSystemError.AddRaw(this, &FGpLoggerCrashManager::OnSystemError);
    FCoreDelegates::ApplicationWillTerminateDelegate.AddRaw(this, &FGpLoggerCrashManager::OnTerminate);
}

void FGpLoggerCrashManager::OnCrashEvent(const FString& Message, const EGpLogLevel Type, const FName& Name, TOptional<double> Time)
{
    TArray<FString> MessageLines;
    Message.ParseIntoArrayLines(MessageLines);
    
    FGpLoggerCrashInternalData CrashData;
    CrashData.LogType = EGpLoggerType::CrashFromUnreal;
    CrashData.LogLevel = Type;
    CrashData.Message = MessageLines.Num() > 0 ? MessageLines[0] : FString();
    CrashData.StackTrace = Message;
    CrashData.DmpData = GpLoggerCrashHelper::Base64Encode(CrashData.StackTrace);
    CrashData.UserFields = {
        { TEXT("Unreal"), FEngineVersion::Current().ToString() }
    };

    GPLOGGER_LOG_DEBUG("%s", *CrashData.DmpData);
    
    CrashDelegate.Broadcast(CrashData);
}

void FGpLoggerCrashManager::OnTerminate()
{
    GPLOGGER_LOG_DEBUG("OnTerminate");
}

void FGpLoggerCrashManager::OnSystemError()
{
    GPLOGGER_LOG_DEBUG("OnHandleSystemError");
}
