#pragma once

#include "GpLoggerInternalCrashTypes.h"
#include "GpLoggerInternalTypes.h"
#include "GpLoggerProtocolBase.h"
#include "Utils/GpLoggerUtils.h"

namespace GpLoggerProtocol
{   
    namespace Payload
    {
        struct FInitializePayload final : FBasePayload
        {
            FString ProjectKey;
            FString ServiceZone;
            bool bEnableCrashReporter;
                
            BEGIN_GPLOGGER_JSON_SERIALIZER
                GPLOGGER_JSON_SERIALIZE("projectKey", ProjectKey);
                GPLOGGER_JSON_SERIALIZE("serviceZone", ServiceZone);
                GPLOGGER_JSON_SERIALIZE("enableCrashReporter", bEnableCrashReporter);
            END_GPLOGGER_JSON_SERIALIZER
        };

        struct FLogPayload final : FBasePayload
        {
            FString Level;
            FString Message;
            TMap<FString, FString> UserFields;
                
            BEGIN_GPLOGGER_JSON_SERIALIZER
                GPLOGGER_JSON_SERIALIZE("level", Level);
                GPLOGGER_JSON_SERIALIZE("message", Message);
                GPLOGGER_JSON_SERIALIZE_MAP("userFields", UserFields);
            END_GPLOGGER_JSON_SERIALIZER
        };

        struct FExceptionPayload final : FBasePayload
        {
            FString Level;
            FString LogType;
            FString Message;
            FString DmpData;
            TMap<FString, FString> UserFields;
                
            BEGIN_GPLOGGER_JSON_SERIALIZER
                GPLOGGER_JSON_SERIALIZE("logLevel", Level);
                GPLOGGER_JSON_SERIALIZE("logType", LogType);
                GPLOGGER_JSON_SERIALIZE("message", Message);
                GPLOGGER_JSON_SERIALIZE("dmpData", DmpData);
                GPLOGGER_JSON_SERIALIZE_MAP("userFields", UserFields);
            END_GPLOGGER_JSON_SERIALIZER
        };
            
        struct FSetLoggerListenerPayload final : FBasePayload
        {
            FString Success;
            FString Filter;
            FString Save;
            FString Error;
                
            BEGIN_GPLOGGER_JSON_SERIALIZER
                GPLOGGER_JSON_SERIALIZE("success", Success);
                GPLOGGER_JSON_SERIALIZE("filter", Filter);
                GPLOGGER_JSON_SERIALIZE("save", Save);
                GPLOGGER_JSON_SERIALIZE("error", Error);
            END_GPLOGGER_JSON_SERIALIZER
        };
    }

    struct FInitializeRequest final : TBridgeProtocol<Payload::FInitializePayload>
    {
        explicit FInitializeRequest(const Payload::FInitializePayload& Payload)
            : TBridgeProtocol(TEXT("toast://logger/initialize"), Payload)
        {
        }
    };
    
    struct FLogRequest final : TBridgeProtocol<Payload::FLogPayload>
    {
        explicit FLogRequest(const Payload::FLogPayload& Payload)
            : TBridgeProtocol(TEXT("toast://logger/log"), Payload)
        {
        }
        
        FLogRequest(const FString& Level, const FString& Message, const TMap<FString, FString>& UserFields)
            : TBridgeProtocol(TEXT("toast://logger/log"))
        {
            Payload.Level = Level;
            Payload.Message = Message;
            Payload.UserFields = UserFields;
        }
    };
    
    struct FSetUserFieldRequest final : TBridgeProtocol<Payload::FSetUserFieldPayload>
    {
        explicit FSetUserFieldRequest(const Payload::FSetUserFieldPayload& Payload)
            : TBridgeProtocol(TEXT("toast://logger/setuserfield"), Payload)
        {
        }
    };
    
    struct FSetLoggerListenerRequest final : TBridgeProtocol<Payload::FSetLoggerListenerPayload>
    {
        FSetLoggerListenerRequest(const FString& SuccessMethod, const FString& FilterMethod, const FString& SaveMethod, const FString& ErrorMethod)
            : TBridgeProtocol(TEXT("toast://logger/setloggerlistener"))
        {
            Payload.Success = SuccessMethod;
            Payload.Filter = FilterMethod;
            Payload.Save = SaveMethod;
            Payload.Error = ErrorMethod;
        }
    };
    
    struct FExceptionRequest final : TBridgeProtocol<Payload::FExceptionPayload>
    {
        explicit FExceptionRequest(const Payload::FExceptionPayload& Payload)
            : TBridgeProtocol(TEXT("toast://logger/exception"), Payload)
        {   
        }
        
        explicit FExceptionRequest(const FGpLoggerCrashInternalData& CrashData)
            : TBridgeProtocol(TEXT("toast://logger/exception"))
        {
            Payload.Level = GpLogger::EnumToString(CrashData.LogLevel).ToUpper();
            Payload.LogType = GpLogger::LogType::GetTypeString(CrashData.LogType);
            Payload.Message = CrashData.Message;
            Payload.DmpData = CrashData.DmpData;
            Payload.UserFields = CrashData.UserFields;
        }
    };
}